;; app_defines.h
;; Specific Defines, 
;; CONSTants and VARiables 
;; of the MIDIBox AY (YM) V2
;; MIOS8 Application -> PIC 18F4685 
;;
;; ================================================================================================
;; RAM (MEMO):
;; The memory space of the 18F4685 is divided into 16 BANKS 
;; that contain 256 bytes each. The data memory contains Special Function Registers (SFRs) 
;; and General Purpose Registers (GPRs). The SFRs are used for control and status of the 
;; controller and peripheral functions, while GPRs are used for data storage and scratchpad 
;; operations in the user's application.
;; 
;; DATA RAM 18F4685 OVERVIEW:
;; ----------------------------------------------------------------------------------------------
;; MIOS8     ACCESS	RAM	=> 0x000 to 0x00F --->	[MIOS8 Special Function Registers  				]
;; MB AY APP ACCESS	RAM	=> 0x010 to 0x05F --->	[ACCESS RAM - FREE FOR MB AY APPLICATION		]
;; SFR (DO NOT USE)		=> 0x060 to 0x0FF ---> 	[PIC SFR's (PIC Special Function Registers)		]
;; MB AY APP BANK  1	=> 0x100 to 0x1FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK  2	=> 0x200 to 0x2FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK  3	=> 0x300 to 0x37F --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MIOS8     BANK  3	=> 0x380 to 0x3FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; MIOS8     BANK  4	=> 0x400 to 0x4FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; MIOS8     BANK  5	=> 0x500 to 0x5FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; MB AY APP BANK  6	=> 0x600 to 0x6FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK  7	=> 0x700 to 0x7FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK  8	=> 0x800 to 0x8FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK  9	=> 0x900 to 0x9FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK 10	=> 0xA00 to 0xAFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK 11	=> 0xB00 to 0xBFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; MB AY APP BANK 12	=> 0xC00 to 0xCFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; SFR (DO NOT USE)		=> 0xD00 to 0xDFF --->	[PIC SFR's (PIC Special Function Registers)		]
;; SFR (DO NOT USE)		=> 0xE00 to 0xEFF --->	[PIC SFR's (PIC Special Function Registers)		]
;; SFR (DO NOT USE)		=> 0xF00 to 0xFFF --->	[PIC SFR's (PIC Special Function Registers)		]
;; ==============================================================================================

;; ==============================================================================================
;; ----------------------------------------------------------------------------------------------
;;                                APPLICATION WORKING REGISTERS
;; ----------------------------------------------------------------------------------------------
;; ==============================================================================================


;; ==============================================================================================
;; ----------------------------------------------------------------------------------------------
;; ---------------[RAM BANK0 | ACESS RAM 0x010 to 0x05F] ----> MIOS8 / PIC 18F4685 <-------------
;; ----------------------------------------------------------------------------------------------
;; ==============================================================================================

;; ==============================================================================================
;; ay_chip_serial_data_interfacing.inc 
;; Shift Register Service Routine CORE ---> 8 Bit Shift Register ---> AY (YM) CHIP 
;; ----------------------------------------------------------------------------------------------
VAR_AY_CHIP_REG_ADDRESS							EQU		0x010 	;; => AY CHIP REG ADRESS BYTE
VAR_AY_CHIP_REG_DATA			        		EQU		0x011 	;; => AY CHIP DATA BYTE
;; ----------------------------------------------------------------------------------------------
;; ----------------------------------------------------------------------------------------------
;; MIDI NOTE ON EVENT VALUE ---> AY CHIP Hz VALUE (TONE A|B|C Tune)
VAR_MIDI_TO_HZ_HIGH_BYTE						EQU   	0x012	; => note_to_hz.inc , etc
VAR_MIDI_TO_HZ_LOW_BYTE							EQU   	0x013	; => note_to_hz.inc , etc
;; ----------------------------------------------------------------------------------------------	
;; ----------------------------------------------------------------------------------------------
;; CLIPBOARD (TEMP)
;; USED FOR TEMP DATA -> OFTEN BY MOVING A BANKED RAM BYTE TO ACCESS RAM 
VAR_TMP_RAM_BYTE_COPY_CLIPBOARD					EQU   	0x014	;; CLIPBOARD   / ZWISCHENABLAGE 
VAR_TMP_RAM_BYTE_COPY_CLIPBOARD2				EQU   	0x015	;; CLIPBOARD 2 / ZWISCHENABLAGE 2
;; ----------------------------------------------------------------------------------------------
VAR_TMP_PATCH_DATA_CLIPBOARD					EQU		0x016 	;; PATCH DATA TMP
																;; USED FOR LOAD/STORE PATCH DATA
																;; FROM PIC EEPROM & BANKSTICK

;; ==============================================================================================
;; --------------------------------[AY-3-8912 SHADOW REGISTERS]----------------------------------
;; ==============================================================================================
;; 
;; NOTE:
;; SEE AY-3-8912 DATASHEETS FOR DETAILS 
VAR_AY_CHIP_REG6_NOISE_PERIOD					EQU		0x017 ;; RANGE 0x00 to 0x1F
VAR_AY_CHIP_REG7_ENABLE_ABC_NOISE_IO			EQU		0x018 ;; RANGE 0x00 to 0xFF
  ;; ----------------[AY CHIP REG 0x07: ASSIGNMENT]-----------------
  ;; >>> | BIT7  | BIT6  | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0 |
  ;; ---------------------------------------------------------------
  ;; >>> | MODE  | MODE  | NOISE| NOISE| NOISE| TONE | TONE | TONE |
  ;; >>> | IOB   | IOA   | CH C | CH B | CH A | CH C | CH B | CH A |
  ;; >>> | /IN   | /IN   | /ON  | /ON  | /ON  | /ON  | /ON  | /ON  |
  ;; >>> | OUT   | OUT   | OFF  | OFF  | OFF  | OFF  | OFF  | OFF  |
  ;; ---------------------------------------------------------------
  ;; HEX.| 0x80  | 0x40  | 0x20 | 0x10 | 0x08 | 0x04 | 0x02 | 0x01 |
  ;; ---------------------------------------------------------------
  ;; DEC.| D128  | D064  | D032 | D016 | D008 | D004 | D002 | D001 |
  ;; ---------------------------------------------------------------

VAR_AY_CHIP_REG8_CH_A_AMPLITUDE		  			EQU		0x019 ;; RANGE 0-15 (0x00 to 0x0F)
  ;; THE VOLUME OF THE CH. A SIGNAL GENERATED BY 
  ;; THE D/A CONVERTER IS DETERMINED BY THE CONTENS OF THE LOWER BITS OF REGISTER 8
  ;; >>> | BIT7 | BIT6 |BIT5|   BIT4   | BIT3 | BIT2 | BIT1 |  BIT0 |
  ;; ----------------------------------------------------------------
  ;; >>> | ---- | ---- |----|    M     | L3   | L2   | L1   | L0    |
  ;; >>> | ---- | ---- |----|ENVELOPE  | VOL. | VOL. | VOL. | VOLUME|
  ;; >>> | ---- | ---- |----|ON/OFF BIT| BIT 3| BIT 2| BIT 1| BIT 0 |
  ;; ----------------------------------------------------------------
  ;; HEX.| ---- | ---- |----|   0x10   | 0x08 | 0x04 | 0x02 | 0x01  |
  ;; ----------------------------------------------------------------
  ;; DEC.| ---- | ---- |----|   D016   | D008 | D004 | D002 | D001  |
  ;; ----------------------------------------------------------------
  ;; VOLUME => 0x00 to 0x0F (DEC. 000 to 015)
  ;; IF BIT4 IS SET => THE ENVELOPE SHAPE/CYCLE IS DEFINED BY REGISTER 13

VAR_AY_CHIP_REG9_CH_B_AMPLITUDE		  			EQU		0x01A ;; RANGE 0-15 (0x00 to 0x0F)
  ;; THE AMPLITUDE OF THE SIGNAS (CH. B) GENERATED BY 
  ;; THE D/A CONVERTER (ONE EACH FOR CHANNELS A, B, C) IS DETERMINED
  ;; BY THE CONTENS OF THE LOWER BITS OF REGISTER 8,9 AND 10 
  ;; >>> | BIT7 | BIT6 |BIT5|   BIT4   | BIT3 | BIT2 | BIT1 |  BIT0 |
  ;; ----------------------------------------------------------------
  ;; >>> | ---- | ---- |----|    M     | L3   | L2   | L1   | L0    |
  ;; >>> | ---- | ---- |----|ENVELOPE  | VOL. | VOL. | VOL. | VOLUME|
  ;; >>> | ---- | ---- |----|ON/OFF BIT| BIT 3| BIT 2| BIT 1| BIT 0 |
  ;; ----------------------------------------------------------------
  ;; HEX.| ---- | ---- |----|   0x10   | 0x08 | 0x04 | 0x02 | 0x01  |
  ;; ----------------------------------------------------------------
  ;; DEC.| ---- | ---- |----|   D016   | D008 | D004 | D002 | D001  |
  ;; ----------------------------------------------------------------
  ;; VOLUME => 0x00 to 0x0F (DEC. 000 to 015)
  ;; IF BIT4 IS SET => THE ENVELOPE SHAPE/CYCLE IS DEFINED BY REGISTER 13

VAR_AY_CHIP_REG10_CH_C_AMPLITUDE		  		EQU		0x01B ;; RANGE 0-15 (0x00 to 0x0F)
  ;; THE AMPLITUDE OF THE SIGNAL (CH. C) GENERATED BY EACH
  ;; OF THE 3 D/A CONVERTERS (ONE EACH FOR CHANNELS A, B, C) IS DETERMINED
  ;; BY THE CONTENS OF THE LOWER BITS OF REGISTER 8,9 AND 10 
  ;; >>> | BIT7 | BIT6 |BIT5|   BIT4   | BIT3 | BIT2 | BIT1 |  BIT0 |
  ;; ----------------------------------------------------------------
  ;; >>> | ---- | ---- |----|    M     | L3   | L2   | L1   | L0    |
  ;; >>> | ---- | ---- |----|ENVELOPE  | VOL. | VOL. | VOL. | VOLUME|
  ;; >>> | ---- | ---- |----|ON/OFF BIT| BIT 3| BIT 2| BIT 1| BIT 0 |
  ;; ----------------------------------------------------------------
  ;; HEX.| ---- | ---- |----|   0x10   | 0x08 | 0x04 | 0x02 | 0x01  |
  ;; ----------------------------------------------------------------
  ;; DEC.| ---- | ---- |----|   D016   | D008 | D004 | D002 | D001  |
  ;; ----------------------------------------------------------------
  ;; VOLUME => 0x00 to 0x0F (DEC. 000 to 015)
  ;; IF BIT4 IS SET => THE ENVELOPE SHAPE/CYCLE IS DEFINED BY REGISTER 13

VAR_AY_CHIP_REG11_ENV_PERIOD_FINE				EQU		0x01C ; RANGE 0x00 to 0xFF
VAR_AY_CHIP_REG12_ENV_PERIOD_COARSE				EQU		0x01D ; RANGE 0x00 to 0xFF
  ;; THE FREQUENCY OF THE ENVELOPE IS OPTAINED BY FIRST COUTING DOWN
  ;; THE INPUT CLOCK BY 256, THEN BY FURTHER COUNTING DOWN THE RESULT
  ;; BY THE PROGRAMMED 16BIT (SEE BIT 0-15) ENVELOPE PERIOD VALUE
  ;; R11 => BIT 0-7
  ;; R12 => BIT 8-15

VAR_AY_CHIP_REG13_ENV_SHAPE_CYCLE				EQU		0x01E ; RANGE 0x00 to 0x0F (D015) 
  ;; THE RELATIVE SHAPE AND CYCLE PATTERN REGISTER
  ;; CH A,B,C ENVELOVE REGISTER IF "AMPLITUDE MODE" *BIT 4*
  ;; @ REGISTER 10 or 11 or 12 (OCTAL 12,13,14) is SET
  ;; FOR DETAILS OF THE GRAPHIC REPRESENTATION SEE AY-3-8912 DATASHEET
  ;;
  ;; >>> | BIT7 | BIT6 |BIT5 | BIT4 | BIT3 | BIT2 | BIT1 |  BIT0 |
  ;; -------------------------------------------------------------
  ;; >>> | ---- | ---- |---- | ---- | ENV. | ENV. | ENV. |  ENV. |
  ;; >>> | ---- | ---- |---- | ---- |  B3  |  B2  |  B1  |  B0   |
  ;; >>> | ---- | ---- |---- | ---- |CONTI-|ATTACK|ALTER-|  B0   |
  ;; >>> | ---- | ---- |---- | ---- |  NUE |      | NATE | HOLD  |
  ;; -------------------------------------------------------------
  ;; HEX.| ---- | ---- |---- | ---- | 0x08 | 0x04 | 0x02 | 0x01  |
  ;; -------------------------------------------------------------
  ;; DEC.| ---- | ---- |---- | ---- | D008 | D004 | D002 | D001  |
  ;; -------------------------------------------------------------

;; ==============================================================================================
;; ---------------------------------[END of AY-3-8912 SHADOW REGISTERS]--------------------------
;; ==============================================================================================


;; ==============================================================================================
;; -----------------------------[FX WORKING REGISTER]--------------------------------------------
VAR_POINTER_TO_CURRENT_FX_TAB_ENTRY								EQU		0x01F
VAR_FX_TAB_ENTRY_CURRENT_FUNCTION								EQU		0x020
VAR_FX_TAB_ENTRY_CURRENT_TIME_PRE_EXECUTION						EQU		0x021
VAR_FX_TAB_ENTRY_CURRENT_PARAM									EQU		0x022 
VAR_Fx_LOAD_WRITE_ADR_COUNTER_TMP								EQU		0x023

;; ==============================================================================================
;; ----------------------------[BUTTON WORKING REGISTER]-----------------------------------------
VAR_CURRENT_DIN_BTN_VALUE						EQU		0x024	
												;; CURRENT BTN VALUE

VAR_CURRENT_DIN_BTN_STATE						EQU		0x025
												;; BUTTON STATE => VALUE 0 = PRESSED
												;; BUTTON STATE => VALUE 1 = DEPRESSED

VAR_BTN_SPECIAL_FLAGS							EQU		0x026
												;; BIT0 = SET 	= BTN IS PERMANENT PRESSED = TRUE
												;; BIT0 = CLEAR	= BTN IS PERMANENT PRESSED = FALSE
												
;; ----------------------------------------------------------------------------------------------
;; user_tick_btn_auto_repeat.inc
VAR_BTN_AUTO_REPEAT_GENERAL_COUNTER				EQU		0x027 	
VAR_FIRST_BTN_STRIKE_CYCLE_COUNTER				EQU		0x028 	
VAR_BTN_AUTO_REPEAT_COUNTER						EQU		0x029 	
;; ----------------------------------------------------------------------------------------------

;; ----------------------------------------------------------------------------------------------
VAR_TONE_NOISE_ABC_USER_ON_OFF_SWITCH			EQU		0x02A 	; FUNCTION: 
																; USER SWITCH TONE/NOISE ABC ON/OFF 
																; FILTER FOR AY REG 7 (NOTE ON / OFF)

;; APPLICATION MAIN MENU WORKING REGISTERS 
;; ==============================================================================================
  
VAR_CURRENT_MAIN_MENU  							EQU		0x02B 	
												;; CURRENT MENU @ LCD

VAR_CURSOR_BLINKING_POS							EQU		0x02C
												;; CURRENT BLINKING CURSOR POS

VAR_CURSOR_BLINKING_CSR_CHAR					EQU		0x02D 	
												;; CURRENT BLINKING CURSOR CHAR @ CURRENT POS

;; ----------------------------------------------------------------------------------------------
;; Envelope Shape Pointer (Env. Shape Help Reg.)
VAR_ENV_SHAPE_0_15_to_1_8_TRANSLATOR			EQU		0x02E
;; ----------------------------------------------------------------------------------------------

VAR_VELOCITY_TO_ABC_NOISE_PERIOD_ON_OFF			EQU   	0x02F 	; SWITCH: Velocity to ABC / NOISE PERIOD
					      						; => midi_evnt_pipeline.inc
					      						; => MENU 7
					      						; => sub_menu_key_minus_menu_7.inc
					      						; => etc.
					     						; BIT 0 = SET => VELOCITY to VOL. TONE A ON
					      						; BIT 1 = SET => VELOCITY to VOL. TONE B ON
					      						; BIT 2 = SET => VELOCITY to VOL. TONE C ON
					      						; BIT 3 = SET => VELOCITY to NOISE PERIOD ON
					      						; BIT 4 - 
					      						; BIT 5 = SET => SWITCH ON VELOCITY TONE A
					      						; BIT 6 = SET => SWITCH ON VELOCITY TONE B
					      						; BIT 7 = SET => SWITCH ON VELOCITY TONE C
;; ----------------------------------------------------------------------------------------------
VAR_MIDI_VELOCITY_TO_AY_CHIP_VOLUME				EQU   	0x030 	; => midi_evnt_pipeline.inc
					      						;; Ergebnis der Umrechnung
					      						;; VON MIDI VELOCITY WERT
					      						;; ZUM VOLUME AY REG WERT
					     						;; 0-127 => 0-15
					      						;; => MIDI_EVNT_VALUE/8
;; ----------------------------------------------------------------------------------------------
VAR_MIDI_VELOCITY_TO_AY_CHIP_NOISE_PERIOD		EQU   	0x031 	; => midi_evnt_pipeline.inc
					      						;; Ergebnis der Umrechnung
					      						;; VON MIDI VELOCITY WERT
					      						;; ZUM NOISE RATE AY REG WERT
					      						;; 0-127 => 0-31
					      						;; => MIDI_EVNT_VALUE/4
;; ----------------------------------------------------------------------------------------------
VAR_AY_CHIP_R6_NOISE_PERIOD_REVERSE_VALUE		EQU     0x032 ; => LABEL_AY_MENU_MIX_NOISE_PERIOD, FX Func
;; ----------------------------------------------------------------------------------------------

;; ------------------------[MIDI RELATED REGISTERS]----------------------------------------------
;; ----------------------------------------------------------------------------------------------
MIDI_EVNT0										EQU		0x033 ;; @ midi_functions.inc
MIDI_EVNT1										EQU		0x034 ;; @ midi_functions.inc
MIDI_EVNT_VALUE									EQU		0x035 ;; @ midi_functions.inc
;; ----------------------------------------------------------------------------------------------
MIDI_CHANNEL_TONE_A_USER                  		EQU   	0x036 ; => midi_functions.inc, main + submenu
MIDI_CHANNEL_TONE_B_USER                  		EQU   	0x037 ; => midi_functions.inc, main + submenu
MIDI_CHANNEL_TONE_C_USER                  		EQU   	0x038 ; => midi_functions.inc, main + submenu
;;-------------------------------------[ CC ]----------------------------------------------------
MIDI_CC_VALUE									EQU		0x039 ;; => midi_controller.inc
;; ----------------------------------------------------------------------------------------------
MIDI_CC_6_TMP									EQU		0x03A ;; => midi_controller.inc
MIDI_CC_7_TMP									EQU		0x03B ;; => midi_controller.inc
MIDI_CC_8_TMP									EQU		0x03C ;; => midi_controller.inc
MIDI_CC_9_TMP									EQU		0x03D ;; => midi_controller.inc
;; ----------------------------------------------------------------------------------------------
VAR_MIDI_EVENT_CC_PB_MODWHEEL_FLAG				EQU		0x03E ;; => user_tick_lcd_redraw.inc
;; ----------------------------------------------------------------------------------------------
;; These Regs are used to check if the last *note off* midi event == the last *note on* 
;; midi event @ Tone a|b|c if so, switch off tone a|b|c if *not* - ignore the note off event
MIDI_LAST_NOTE_ON_EVENT_VALUE_TONE_A			EQU   	0x03F 	;; => note_on.inc, note_off.inc
MIDI_LAST_NOTE_ON_EVENT_VALUE_TONE_B			EQU   	0x040 	;; => note_on.inc, note_off.inc
MIDI_LAST_NOTE_ON_EVENT_VALUE_TONE_C			EQU   	0x041 	;; => note_on.inc, note_off.inc
;; ----------------------------------------------------------------------------------------------
MIDI_CURRENT_EVNT_CHANNEL						EQU		0x042 	;; CURRENT MIDI EVENT CHANNEL 
;; ----------------------------------------------------------------------------------------------
MIDI_TONE_ABC_ON_OFF							EQU		0x043 	;; FOR MIDI NOTE ON NOTE OFF SWITCHING
;; ----------------------------------------------------------------------------------------------
VAR_TONE_A_TUNE_FINE_USER                     	EQU   	0x044 	;
VAR_TONE_B_TUNE_FINE_USER                     	EQU   	0x045 	; 
VAR_TONE_C_TUNE_FINE_USER                     	EQU   	0x046 	; 
VAR_TONE_A_TUNE_COARSE_USER                    	EQU   	0x047 	; 
VAR_TONE_B_TUNE_COARSE_USER                    	EQU   	0x048 	; 
VAR_TONE_C_TUNE_COARSE_USER                    	EQU   	0x049 	; 
;; ----------------------------------------------------------------------------------------------
VAR_TONE_A_TUNE_FINE_MIDI                     	EQU   	0x04A 	; 
VAR_TONE_B_TUNE_FINE_MIDI                     	EQU   	0x04B 	; 
VAR_TONE_C_TUNE_FINE_MIDI                     	EQU   	0x04C 	; 
VAR_TONE_A_TUNE_COARSE_MIDI                    	EQU   	0x04D 	; 
VAR_TONE_B_TUNE_COARSE_MIDI                    	EQU   	0x04E 	; 
VAR_TONE_C_TUNE_COARSE_MIDI                    	EQU   	0x04F 	; 
VAR_AY_DETUNE_TEMP								EQU   	0x050	; 
;; ----------------------------------------------------------------------------------------------
VAR_TONE_A_TUNE_PITCH_BEND						EQU		0x051 	;; => midi_event_pitch_wheel.inc
VAR_TONE_B_TUNE_PITCH_BEND						EQU		0x052 	;; => midi_event_pitch_wheel.inc
VAR_TONE_C_TUNE_PITCH_BEND						EQU		0x053 	;; => midi_event_pitch_wheel.inc
VAR_TONE_A_TUNE_COARSE_MIDI_PITCH_BEND_TMP		EQU 	0x054 	;; => midi_event_pitch_wheel.inc
VAR_TONE_B_TUNE_COARSE_MIDI_PITCH_BEND_TMP		EQU 	0x055 	;; => midi_event_pitch_wheel.inc
VAR_TONE_C_TUNE_COARSE_MIDI_PITCH_BEND_TMP		EQU 	0x056 	;; => midi_event_pitch_wheel.inc
VAR_TONE_A_TUNE_FINE_MIDI_PITCH_BEND_TMP		EQU 	0x057 	;; => midi_event_pitch_wheel.inc
VAR_TONE_B_TUNE_FINE_MIDI_PITCH_BEND_TMP		EQU 	0x058	;; => midi_event_pitch_wheel.inc
VAR_TONE_C_TUNE_FINE_MIDI_PITCH_BEND_TMP		EQU 	0x059	;; => midi_event_pitch_wheel.inc
VAR_TMP6										EQU		0x05A	;; => sub_routine_button_shortcuts_menu8.inc
VAR_TMP7										EQU		0x05B	;; => midi_evnt_handling.inc :: MIDI_EVNT_IS_Cx
VAR_TMP8										EQU		0x05C	;; FREE RAM BYTE (NOT IN USE YET)
TMP9											EQU		0x05D	;; FREE RAM BYTE (NOT IN USE YET)
TMP10											EQU		0x05E	;; FREE RAM BYTE (NOT IN USE YET)
TMP11											EQU		0x05F	;; FREE RAM BYTE (NOT IN USE YET)

;; ==============================================================================================
;; =================================[END: ACCESS RAM AREA]=======================================
;; ==============================================================================================


;; ==============================================================================================
;; [SFR (DO NOT USE) =====> 0x060 to 0x0FF ---> [PIC SFR's (PIC Special Function Registers)		] 
;; ==============================================================================================


;; ==============================================================================================
;; MB AY APP DATA RAM BANK 1 => 0x100 to 0x1FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION	]
;; ==============================================================================================
;; =============================================[START: RAM BANK1]===============================
;; ==============================================================================================
;; ---------------------------------------------[BSR = 1         ]-------------------------------
;; ----------------------------------------------------------------------------------------------
;; CURSOR POS @ MENU [0-8] on LCD 
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_CHOICE_MENU					EQU 0x100	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_PATCH_INFO					EQU 0x101	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_ABC_TUNE						EQU 0x102	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_ABC_NOISE						EQU 0x103	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_AMPLITUDE						EQU 0x104	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_ENVELOPE						EQU 0x105	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_MIDI_CONFIG					EQU	0x106	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_VELOCITY_CONFIG				EQU	0x107	
BNKD_VAR_LCD_CURSOR_POS_MAIN_MENU_FX							EQU	0x108

;; CURSOR BLINKING 2x 16BIT COUNTER
;; USED BY MIOS_HLP_16bitAddSaturate
;; NOTE: DO NOT CHANGE BECAUSE OF INDIRECT READING AND WRITING 
#define 	CONST_BSR1_SELECT_RAM_BANK1							0x01	
#define 	CONST_RAM_ADR_OFFSET_CSR_16BIT_TIMER1				0x10
BNKD_VAR_CRS_BLINKING_TIMER1L									EQU	0x110
BNKD_VAR_CRS_BLINKING_TIMER1H									EQU	0x111
#define 	CONST_RAM_ADR_OFFSET_CSR_16BIT_TIMER2				0x12
BNKD_VAR_CRS_BLINKING_TIMER2L									EQU	0x112
BNKD_VAR_CRS_BLINKING_TIMER2H									EQU	0x113
;; BLINKING CSR TRIGGER 
BNKD_VAR_CRS_BLINKING_TRIGGER									EQU	0x114



;; ==============================================================================================
;; MB AY APP BANK 2	=> 0x200 to 0x2FF --->		[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================
;; =============================================[START: RAM BANK2]===============================
;; ---------------------------------------------[BSR = 2         ]-------------------------------
;; ==============================================================================================

;; Fx Option Const
;; PIC Fx RAM AREA 		: 0x220 to  0x280 (DEC 544-640)
;; Fx EEPROM AREA		: 0x20  to  0x80
;; Fx BANKSTICK AREA    : 0x1D  to  0x7C
;; -----------------------------------------------------------------------------------------------
#define CONST_BSR2_SELECT_Fx_RAM_BANK							0x02	;; Fx DATA IS LOCATED @ RAM BANK 2
;; --------------------------
#define CONST_RAM_ADR_OFFSET_FX_FUNCTION						0x20	;; PIC RAM PATCH: Fx Func. Area OFFSET
#define CONST_EEPROM_OFFSET_ADR_FX_FUNCTION						0x20	;; EEPROM INT PATCH: Fx Func. Area OFFSET
#define CONST_BANKSTICK_PATCH_OFFSET_FX_FUNCTION				0x1D	;; BANKSTICK PATCH : Fx Func. Area OFFSET
;; --------------------------
#define RAM_OFFSET_FX_DATA										0x40	;; PIC RAM PATCH: Fx DATA Area OFFSET
#define EEPROM_OFFSET_FX_DATA									0x40	;; PIC EEPROM INT PATCH: Fx DATA Area OFFSET
#define BANKSTICK_OFFSET_FX_DATA								0x3D	;; BANKSTICK PATCH     : Fx DATA Area OFFSET
;; --------------------------
#define RAM_OFFSET_FX_TIME_PRE_EXECUTION						0x60	;; PIC RAM PATCH: Fx TIME Area OFFSET
#define EEPROM_OFFSET_FX_TIME_PRE_EXECUTION						0x60	;; PIC EEPROM INT PATCH: Fx TIME Area OFFSET
#define BANKSTICK_OFFSET_FX_TIME_PRE_EXECUTION					0x5D	;; BANKSTICK PATCH     : Fx TIME Area OFFSET
;; --------------------------
#define FX_POS_MAX_31											0x1F 	;; 31 dec.(=> fx_pipeline.inc)
#define FX_POS_MAX_PLUS1										0x20 	;; Fx Pos Max +1 = 32
#define Fx_DELAY_TIME_MAX										0xC8 	;; 200 dec
#define FX_FUNCTIONS_MAX										0x29 	;; MAX Number of Fx Functions
#define FX_FUNCTIONS_MAX_PULS_1									0x2A 	;; MAX Number of Fx Functions +1
#define FX_POS_MAX												0x1F 	;; Fx Pos Max = 31



;; ----------------------------[Fx Function and Function data Registers]--------------------------
;; -----------------------------------------------------------------------------------------------
;; NOTE: DO NOT CHANGE BECAUSE OF INDIRECT READING AND WRITING 
BNKD_VAR_FX_FUNCTION_POS0									EQU	0x220
BNKD_VAR_FX_FUNCTION_POS1									EQU	0x221
BNKD_VAR_FX_FUNCTION_POS2									EQU	0x222
BNKD_VAR_FX_FUNCTION_POS3									EQU	0x223
BNKD_VAR_FX_FUNCTION_POS4									EQU	0x224
BNKD_VAR_FX_FUNCTION_POS5									EQU	0x225
BNKD_VAR_FX_FUNCTION_POS6									EQU	0x226
BNKD_VAR_FX_FUNCTION_POS7									EQU	0x227
BNKD_VAR_FX_FUNCTION_POS8									EQU	0x228
BNKD_VAR_FX_FUNCTION_POS9									EQU	0x229
BNKD_VAR_FX_FUNCTION_POS10									EQU	0x22A
BNKD_VAR_FX_FUNCTION_POS11									EQU	0x22B
BNKD_VAR_FX_FUNCTION_POS12									EQU	0x22C
BNKD_VAR_FX_FUNCTION_POS13									EQU	0x22D
BNKD_VAR_FX_FUNCTION_POS14									EQU	0x22E
BNKD_VAR_FX_FUNCTION_POS15									EQU	0x22F
BNKD_VAR_FX_FUNCTION_POS16									EQU	0x230
BNKD_VAR_FX_FUNCTION_POS17									EQU	0x231
BNKD_VAR_FX_FUNCTION_POS18									EQU	0x232
BNKD_VAR_FX_FUNCTION_POS19									EQU	0x233
BNKD_VAR_FX_FUNCTION_POS20									EQU	0x234
BNKD_VAR_FX_FUNCTION_POS21									EQU	0x235
BNKD_VAR_FX_FUNCTION_POS22									EQU	0x236
BNKD_VAR_FX_FUNCTION_POS23									EQU	0x237
BNKD_VAR_FX_FUNCTION_POS24									EQU	0x238
BNKD_VAR_FX_FUNCTION_POS25									EQU	0x239
BNKD_VAR_FX_FUNCTION_POS26									EQU	0x23A
BNKD_VAR_FX_FUNCTION_POS27									EQU	0x23B
BNKD_VAR_FX_FUNCTION_POS28									EQU	0x23C
BNKD_VAR_FX_FUNCTION_POS29									EQU	0x23D
BNKD_VAR_FX_FUNCTION_POS30									EQU	0x23E
BNKD_VAR_FX_FUNCTION_POS31									EQU	0x23F

BNKD_VAR_FX_FUNCTION_DATA_POS0								EQU	0x240
BNKD_VAR_FX_FUNCTION_DATA_POS1								EQU	0x241
BNKD_VAR_FX_FUNCTION_DATA_POS2								EQU	0x242
BNKD_VAR_FX_FUNCTION_DATA_POS3								EQU	0x243
BNKD_VAR_FX_FUNCTION_DATA_POS4								EQU	0x244
BNKD_VAR_FX_FUNCTION_DATA_POS5								EQU	0x245
BNKD_VAR_FX_FUNCTION_DATA_POS6								EQU	0x246
BNKD_VAR_FX_FUNCTION_DATA_POS7								EQU	0x247
BNKD_VAR_FX_FUNCTION_DATA_POS8								EQU	0x248
BNKD_VAR_FX_FUNCTION_DATA_POS9								EQU	0x249
BNKD_VAR_FX_FUNCTION_DATA_POS10								EQU	0x24A
BNKD_VAR_FX_FUNCTION_DATA_POS11								EQU	0x24B
BNKD_VAR_FX_FUNCTION_DATA_POS12								EQU	0x24C
BNKD_VAR_FX_FUNCTION_DATA_POS13								EQU	0x24D
BNKD_VAR_FX_FUNCTION_DATA_POS14								EQU	0x24E
BNKD_VAR_FX_FUNCTION_DATA_POS15								EQU	0x24F
BNKD_VAR_FX_FUNCTION_DATA_POS16								EQU	0x250
BNKD_VAR_FX_FUNCTION_DATA_POS17								EQU	0x251
BNKD_VAR_FX_FUNCTION_DATA_POS18								EQU	0x252
BNKD_VAR_FX_FUNCTION_DATA_POS19								EQU	0x253
BNKD_VAR_FX_FUNCTION_DATA_POS20								EQU	0x254
BNKD_VAR_FX_FUNCTION_DATA_POS21								EQU	0x255
BNKD_VAR_FX_FUNCTION_DATA_POS22								EQU	0x256
BNKD_VAR_FX_FUNCTION_DATA_POS23								EQU	0x257
BNKD_VAR_FX_FUNCTION_DATA_POS24								EQU	0x258
BNKD_VAR_FX_FUNCTION_DATA_POS25								EQU	0x259
BNKD_VAR_FX_FUNCTION_DATA_POS26								EQU	0x25A
BNKD_VAR_FX_FUNCTION_DATA_POS27								EQU	0x25B
BNKD_VAR_FX_FUNCTION_DATA_POS28								EQU	0x25C
BNKD_VAR_FX_FUNCTION_DATA_POS29								EQU	0x25D
BNKD_VAR_FX_FUNCTION_DATA_POS30								EQU	0x25E
BNKD_VAR_FX_FUNCTION_DATA_POS31								EQU	0x25F

BNKD_VAR_FX_FUNCTION_TIME_POS0								EQU	0x260
BNKD_VAR_FX_FUNCTION_TIME_POS1								EQU	0x261
BNKD_VAR_FX_FUNCTION_TIME_POS2								EQU	0x262
BNKD_VAR_FX_FUNCTION_TIME_POS3								EQU	0x263
BNKD_VAR_FX_FUNCTION_TIME_POS4								EQU	0x264
BNKD_VAR_FX_FUNCTION_TIME_POS5								EQU	0x265
BNKD_VAR_FX_FUNCTION_TIME_POS6								EQU	0x266
BNKD_VAR_FX_FUNCTION_TIME_POS7								EQU	0x267
BNKD_VAR_FX_FUNCTION_TIME_POS8								EQU	0x268
BNKD_VAR_FX_FUNCTION_TIME_POS9								EQU	0x269
BNKD_VAR_FX_FUNCTION_TIME_POS10								EQU	0x26A
BNKD_VAR_FX_FUNCTION_TIME_POS11								EQU	0x26B
BNKD_VAR_FX_FUNCTION_TIME_POS12								EQU	0x26C
BNKD_VAR_FX_FUNCTION_TIME_POS13								EQU	0x26D
BNKD_VAR_FX_FUNCTION_TIME_POS14								EQU	0x26E
BNKD_VAR_FX_FUNCTION_TIME_POS15								EQU	0x26F
BNKD_VAR_FX_FUNCTION_TIME_POS16								EQU	0x270
BNKD_VAR_FX_FUNCTION_TIME_POS17								EQU	0x271
BNKD_VAR_FX_FUNCTION_TIME_POS18								EQU	0x272
BNKD_VAR_FX_FUNCTION_TIME_POS19								EQU	0x273
BNKD_VAR_FX_FUNCTION_TIME_POS20								EQU	0x274
BNKD_VAR_FX_FUNCTION_TIME_POS21								EQU	0x275
BNKD_VAR_FX_FUNCTION_TIME_POS22								EQU	0x276
BNKD_VAR_FX_FUNCTION_TIME_POS23								EQU	0x277
BNKD_VAR_FX_FUNCTION_TIME_POS24								EQU	0x278
BNKD_VAR_FX_FUNCTION_TIME_POS25								EQU	0x279
BNKD_VAR_FX_FUNCTION_TIME_POS26								EQU	0x27A
BNKD_VAR_FX_FUNCTION_TIME_POS27								EQU	0x27B
BNKD_VAR_FX_FUNCTION_TIME_POS28								EQU	0x27C
BNKD_VAR_FX_FUNCTION_TIME_POS29								EQU	0x27D
BNKD_VAR_FX_FUNCTION_TIME_POS30								EQU	0x27E
BNKD_VAR_FX_FUNCTION_TIME_POS31								EQU	0x27F


;; ==============================================================================================
;; MB AY APP BANK  3	=> 0x300 to 0x37F --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================
;; =============================================[START: RAM BANK3]===============================
;; ==============================================================================================
;; ---------------------------------------------[BSR = 3         ]-------------------------------
;; ==============================================================================================

;; ---------------------------------------------[PATCH RELATED]----------------------------------
BNKD_VAR_CURRENT_ASC_VALUE_PATCH_NAME			EQU		0x300	;; FOR NAME EDITING
BNKD_VAR_POS_CURSOR_PATCH_NAME					EQU		0x301	;; FOR NAME EDITING etc.
																;; OF THE PATCH NAME

BNKD_VAR_PATCH_NAME_ASC_0						EQU		0x303	;; CURRENT PATCH NAME ASC VALUE0
BNKD_VAR_PATCH_NAME_ASC_1						EQU		0x304	;; CURRENT PATCH NAME ASC VALUE1
BNKD_VAR_PATCH_NAME_ASC_2						EQU		0x305	;; CURRENT PATCH NAME ASC VALUE2
BNKD_VAR_PATCH_NAME_ASC_3						EQU		0x306	;; CURRENT PATCH NAME ASC VALUE3
BNKD_VAR_PATCH_NAME_ASC_4						EQU		0x307	;; CURRENT PATCH NAME ASC VALUE4
BNKD_VAR_PATCH_NAME_ASC_5						EQU		0x308	;; CURRENT PATCH NAME ASC VALUE5
BNKD_VAR_PATCH_NAME_ASC_6						EQU		0x309	;; CURRENT PATCH NAME ASC VALUE6
BNKD_VAR_PATCH_NAME_ASC_7						EQU		0x30A	;; CURRENT PATCH NAME ASC VALUE7

;; ---------------------------[PATCH NUMBER (BANKSTICK, PIC EEPROM)]-----------------------------
;; PATCH MENU (MAIN MENU 1)
																;; LOADED OR WRITTEN PATCH
BNKD_VAR_CURRENT_PATCH_NR						EQU		0x30B 	;; 0 = INT (PIC EEPROM PATCH)
																;; 1 - 128 (BANK STICK PATCHES)

BNKD_VAR_BROWSE_PATCH_NR						EQU		0x30C	;; FOR BROWSING PATCHES

BNKD_VAR_PATCH_STATE_FLAGS						EQU		0x30D 	;; PATCH STATE FLAGS 
																;; SEE CONST AREA FOR DETAILS

BNKD_VAR_FORMAT_BANKSTICK_CYCLE_COUNTER			EQU		0x30E	;; FOR BS FORMATTING
BNKD_VAR_BANKSTICK_LOW_BYTE_ADR_COUNTER			EQU		0x30F 	;; BS LOW BYTE ADR 


;; ==============================================================================================
;; MIOS8     BANK  3	=> 0x380 to 0x3FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; MIOS8     BANK  4	=> 0x400 to 0x4FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; MIOS8     BANK  5	=> 0x500 to 0x5FF --->	[BANKED RAM - MIOS8 Special Function Registers	]
;; ==============================================================================================


;; ==============================================================================================
;; MB AY APP BANK  6	=> 0x600 to 0x6FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================
;;-----------------------------------------------------------------------------------------------
;; ----------------------------[Fx REGISTER]----------------------------------------------------- 

BNKD_VAR_FX_USER_ON_OFF_ABC								EQU 	0x600
														;; ON OFF FX @ TONE A|B|C
														;; BIT 0 => FX ON (SET) @ TONE A
														;; BIT 1 => FX ON (SET) @ TONE B
														;; BIT 2 => FX ON (SET) @ TONE C

BNKD_VAR_Fx_MIDI_NOTE_ON_OR_OFF							EQU 	0x601
														;; BIT 0 => MIDI NOTE ON (SET) @ TONE A
														;; BIT 1 => MIDI NOTE ON (SET) @ TONE B
														;; BIT 2 => MIDI NOTE ON (SET) @ TONE C
														;; BIT 3 => Reserved
														;; BIT 4 => Reserved
														;; BIT 5 => Reserved
														;; BIT 6 => Reserved
														;; BIT 7 => Reserved

BNKD_VAR_FX_POS_CURRENT									EQU 	0x602
														;; CURRENT POS OF Fx Table (ROW)
														;; Fx Tab ROW Pointer

BNKD_VAR_FX_FUNCTION_CURRENT							EQU 	0x603
														;; Fx FUNCTIONS
														;; Fx = 0 => DO x
														;; FX = 1 => DO y
														;; FX = 2 => DO z
														;; etc.
						
BNKD_VAR_FX_DATA_CURRENT								EQU 	0x604
														;; USED for WRITING DATA TO AN AY CHIP REG.
														;; SOME FX FUNCTIONS NEED DATA
						

BNKD_VAR_FX_TIME_PRE_EXECUTION_CURRENT					EQU		0x605
														;; TIME PERIOD TO THE NEXT FX POS.
														;; EXECUTION

BNKD_VAR_FX_TIMER										EQU 	0x606
														;; Triggert (inc +1) by timer0 overflow

BNKD_VAR_FX_MIDI_NOTE_ON_SESSION						EQU		0x607
														;; EVERY NOTE ON @ [A|B|C] GENERATES A NEW
														;; SESSION VALUE NUMBER (+1 @ BNKD_VAR_FX_PIPELINE_SESSION_NUMBER) 
														;; FOR A NEW FX SESSION. EVERY TIME THERE IS
														;; A NEW FX SESSION NUMBER THE "FX POS"
														;; WILL BE RESET TO 1 (A RESET OF FX POS WILL TAKE PLACE)

BNKD_VAR_FX_PIPELINE_SESSION_NUMBER						EQU		0x608
														;; Fx SESSION VALUE NUMBER	

BNKD_VAR_STATE_FX_NOP_MODE								EQU		0x609
														;; = 0 = DO FX OPERATION 	(FX NOP MODE OFF)
														;; > 0 = DO NO FX OPERATION (FX NOP MODE ON )

BNKD_VAR_Fx_TEMP										EQU		0x60A

BNKD_VAR_FX_NOTE_ON_1_TUNE_FINE							EQU		0x60B
BNKD_VAR_FX_NOTE_ON_1_TUNE_COARSE						EQU		0x60C

BNKD_VAR_FX_NOTE_ON_2_TUNE_FINE							EQU		0x60D
BNKD_VAR_FX_NOTE_ON_2_TUNE_COARSE						EQU		0x60E

BNKD_VAR_FX_NOTE_ON_3_TUNE_FINE							EQU		0x60F
BNKD_VAR_FX_NOTE_ON_3_TUNE_COARSE						EQU		0x610

;; ----------------------------------------------------------------------------------------------
BNKD_FX_TUNE_COARSE_TEMP								EQU		0x611	; Fx Call for +/- Tune
BNKD_FX_TUNE_FINE_TEMP									EQU		0x612	; Fx Call for +/- Tune
;; ----------------------------------------------------------------------------------------------	


;; ==============================================================================================
;; MB AY APP BANK  7	=> 0x700 to 0x7FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================


;; ==============================================================================================
;; MB AY APP BANK  8	=> 0x800 to 0x8FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================

;; ==============================================================================================
;; MB AY APP BANK  9	=> 0x900 to 0x9FF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================

;; ==============================================================================================
;; MB AY APP BANK 10	=> 0xA00 to 0xAFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================

;; ==============================================================================================
;; MB AY APP BANK 11	=> 0xB00 to 0xBFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================

;; ==============================================================================================
;; MB AY APP BANK 12	=> 0xC00 to 0xCFF --->	[BANKED RAM - FREE FOR MB AY APPLICATION		]
;; ==============================================================================================


;; ----------------------------------------------------------------------------------------------
;; --------------------------------------[RAM END]-----------------------------------------------
;; ----------------------------------------------------------------------------------------------


;; --------------------------------------[constants]---------------------------------------------
;; ##############################################################################################
;; ##############################################################################################
;; #                                     MIDIBox AY APPLICATION CONST DEF                       #
;; ##############################################################################################
;; ##############################################################################################


;; ##############################################################################################
;; # DEF POS 0 FOR LCD ROW 1 to 4 @ LCD(4x16)                                                   #
;; ##############################################################################################
#define LCD_ROW1 											0x00 	; Pos.0 @ LCD Row1 (4x16) 
#define LCD_ROW2 											0x40	; Pos.0 @ LCD Row2 (4x16) 
#define LCD_ROW3 											0x10	; Pos.0 @ LCD Row3 (4x16) 
#define LCD_ROW4 											0x50	; Pos.0 @ LCD Row4 (4x16)

;; ##############################################################################################
;; # DEF POS 0 FOR LCD ROW 1 to 4 @ LCD(4x20)                                                   #
;; ##############################################################################################
;; #define LCD_ROW1 										0x00 	; Pos.0 @ LCD Row1 (4x20)
;; #define LCD_ROW2 										0x40	; Pos.0 @ LCD Row2 (4x20)
;; #define LCD_ROW3 										0x80	; Pos.0 @ LCD Row3 (4x20)
;; #define LCD_ROW4 										0xC0	; Pos.0 @ LCD Row4 (4x20)

;; MAIN MENU 
;; -----------------------------------------------------------------------------------------------
#define CONST_INI_START_UP_MAIN_MENU						0x00	;; SHOW MAIN MENU 0 after Startup
#define CONST_MAIN_MENU_00									0x00	;; MAIN MENU 0 (SELECT MENU)
#define CONST_MAIN_MENU_01									0x01	;; PATCH LOAD / WRITE
#define CONST_MAIN_MENU_02									0x02	;; ABC (DE) TUNE
#define CONST_MAIN_MENU_03									0x03	;; TONE/NOISE/FX ABC - MIXER
#define CONST_MAIN_MENU_04									0x04	;; TONE ABC AMPLITUDE
#define CONST_MAIN_MENU_05									0x05	;; ENVELOPE
#define CONST_MAIN_MENU_06									0x06	;; MIDI CHANNEL CONFIG
#define CONST_MAIN_MENU_07									0x07	;; VELOCITY CONFIG
#define CONST_MAIN_MENU_08									0x08	;; Fx EDITOR


;; MAIN MENU COURSER POS (SELECTED ITEM)
;; -----------------------------------------------------------------------------------------------
#define CONST_MENU_ITEM_POS0_NOT_VALID						0x00
#define CONST_MENU_ITEM_POS1								0x01
#define CONST_MENU_ITEM_POS2								0x02
#define CONST_MENU_ITEM_POS3								0x03
#define CONST_MENU_ITEM_POS4								0x04
#define CONST_MENU_ITEM_POS5								0x05
#define CONST_MENU_ITEM_POS6								0x06
#define CONST_MENU_ITEM_POS7								0x07
#define CONST_MENU_ITEM_POS8								0x08

;; BTN STATE CONST
#define CONST_PRESSED_BTN_STATE_VALUE0						0x00

;; USED BY BLINKING CURSOR FUNCTION
;; BLINKING CURSOR POS FOR -> VAR_CURSOR_BLINKING_POS
;; MENU 0
#define CSR_POS_MENU0_ITEM1							 	LCD_ROW1 
#define CSR_POS_MENU0_ITEM2								LCD_ROW2 
#define CSR_POS_MENU0_ITEM3								LCD_ROW3 
#define CSR_POS_MENU0_ITEM4								LCD_ROW4 
#define CSR_POS_MENU0_ITEM5								LCD_ROW1 + 10
#define CSR_POS_MENU0_ITEM6								LCD_ROW2 + 10
#define CSR_POS_MENU0_ITEM7								LCD_ROW3 + 10
#define CSR_POS_MENU0_ITEM8								LCD_ROW4 + 10
;; LCD CSR POS MENU 1
#define CSR_POS_MENU1_ITEM1								LCD_ROW1 + 5
#define CSR_POS_MENU1_ITEM2								LCD_ROW2 + 5
;; LCD CSR POS MENU 2
#define CSR_POS_MENU2_ITEM1								LCD_ROW2 + 5
#define CSR_POS_MENU2_ITEM2								LCD_ROW2 + 11
#define CSR_POS_MENU2_ITEM3								LCD_ROW3 + 5
#define CSR_POS_MENU2_ITEM4								LCD_ROW3 + 11
#define CSR_POS_MENU2_ITEM5								LCD_ROW4 + 5
#define CSR_POS_MENU2_ITEM6								LCD_ROW4 + 11
;; LCD CSR POS MENU 3
#define CSR_POS_MENU3_ITEM1								LCD_ROW1		;; NOT IN USE
#define CSR_POS_MENU3_ITEM2								LCD_ROW2 		;; NOT IN USE
#define CSR_POS_MENU3_ITEM3								LCD_ROW3 + 11	
#define CSR_POS_MENU3_ITEM4								LCD_ROW4		;; NOT IN USE
;; LCD CSR POS MENU 4
#define CSR_POS_MENU4_ITEM1								LCD_ROW2 + 7
#define CSR_POS_MENU4_ITEM2								LCD_ROW3 + 7
#define CSR_POS_MENU4_ITEM3								LCD_ROW4 + 7
;; LCD CSR POS MENU 5
#define CSR_POS_MENU5_ITEM1								LCD_ROW1 + 11
#define CSR_POS_MENU5_ITEM2								LCD_ROW2 + 11
#define CSR_POS_MENU5_ITEM3								LCD_ROW3 + 6
#define CSR_POS_MENU5_ITEM4								LCD_ROW4 + 11
;; LCD CSR POS MENU 6
#define CSR_POS_MENU6_ITEM1								LCD_ROW2 + 12
#define CSR_POS_MENU6_ITEM2								LCD_ROW3 + 12
#define CSR_POS_MENU6_ITEM3								LCD_ROW4 + 12
;; LCD CSR POS MENU 7
#define CSR_POS_MENU7_ITEM1								LCD_ROW1 + 12
#define CSR_POS_MENU7_ITEM2								LCD_ROW2 + 12
#define CSR_POS_MENU7_ITEM3								LCD_ROW3 + 12
#define CSR_POS_MENU7_ITEM4								LCD_ROW4 + 12
;; LCD CSR POS MENU 8
#define CSR_POS_MENU8_ITEM1								LCD_ROW1 + 12
#define CSR_POS_MENU8_ITEM2								LCD_ROW2 + 12
#define CSR_POS_MENU8_ITEM3								LCD_ROW3 + 3
#define CSR_POS_MENU8_ITEM4								LCD_ROW4 + 12

;; LCD START SCREEN TIME (STARTUP)
#define START_SCREEN_TIME 								0x10

;; MAIN MENU - COURSER POS INI
;; -----------------------------------------------------------------------------------------------
#define INI_COURSER_POS									0x01 	;; (Do not Change, POS 0 is NOT VALID!)


;; INI VALUES FOR AY CHIP SHADOW REGISTER
#define INI_REG0_FINE_TUNE_A							0x40
#define INI_REG1_COARSE_TUNE_A							0x01
#define INI_REG2_FINE_TUNE_B							0x42
#define INI_REG3_COARSE_TUNE_B							0x09
#define INI_REG4_FINE_TUNE_C							0x44
#define INI_REG5_COARSE_TUNE_C							0x02
#define INI_REG6_NOISE_PERIOD							0x04
#define INI_REG7_TONE_ABC_NOISE_ABC_IO_AB				0xF8
#define INI_REG8_VOLUME_A_ENVELOPE_ON_OFF				0x0F	;; INI VOLUME TONE A	
#define INI_REG9_VOLUME_B_ENVELOPE_ON_OFF				0x0F	;; INI VOLUME TONE B	
#define INI_REG10_VOLUME_C_ENVELOPE_ON_OFF				0x0F	;; INI VOLUME TONE C	
#define INI_REG11_ENVELOPE_FINE							0x0F	;; DEC 15 [0-255]
#define INI_REG12_ENVELOPE_COARSE						0x0F	;; DEC 15 [0-255]
#define INI_REG13_ENVELOPE_SHAPE						0x01	;; ENV. SHAPE = 1
#define INI_REG14_IO_A									0xAA 	;; IO A NOT IN USE YET
#define INI_REG15_IO_B									0x00	;; IO B NOT IN USE 



;; SET *ADR* CONST FOR AY CHIP SHADOW REGISTER 
;; ----------------------------------------------------------------------------------------------
#define CONST_AY_CHIP_ADR_00_TONE_A_FREQUENCY_FINE	 			0x00
#define CONST_AY_CHIP_ADR_01_TONE_A_FREQUENCY_PERIOD_COARSE 	0x01
#define CONST_AY_CHIP_ADR_02_TONE_B_FREQUENCY_PERIOD_FINE		0x02
#define CONST_AY_CHIP_ADR_03_TONE_B_FREQUENCY_PERIOD_COARSE 	0x03
#define CONST_AY_CHIP_ADR_04_TONE_C_FREQUENCY_PERIOD_FINE		0x04
#define CONST_AY_CHIP_ADR_05_TONE_C_TONE_PERIOD_COARSE			0x05
#define CONST_AY_CHIP_ADR_06_NOISE_PERIOD						0x06
#define CONST_AY_CHIP_ADR_07_TONE_NOISE_ON_OFF					0x07
#define CONST_AY_CHIP_ADR_08_TONE_A_VOLUME_ENV_MODE_ON_OFF		0x08
#define CONST_AY_CHIP_ADR_09_TONE_B_VOLUME_ENV_MODE_ON_OFF		0x09
#define CONST_AY_CHIP_ADR_10_TONE_C_VOLUME_ENV_MODE_ON_OFF		0x0A
#define CONST_AY_CHIP_ADR_11_GLOBAL_ENVELOPE_FINE				0x0B
#define CONST_AY_CHIP_ADR_12_GLOBAL_ENVELOPE_COARSE				0x0C
#define CONST_AY_CHIP_ADR_13_ENVELOPE_SHAPE_AND_CYCLE			0x0D
#define CONST_AY_CHIP_ADR_14_IO_PORT_A_DATA						0x0E
#define CONST_AY_CHIP_ADR_15_IO_PORT_B_DATA						0x0F




;; TONE ABC ON/OFF
;; NOISE ABC ON/OFF
;; -----------------------------------------------------------------------------------------------
#define BIT0_TONE_A_ON_OFF									0
#define BIT1_TONE_B_ON_OFF									1
#define BIT2_TONE_C_ON_OFF									2	
#define BIT3_NOISE_A_ON_OFF									3
#define BIT4_NOISE_B_ON_OFF									4
#define BIT5_NOISE_C_ON_OFF									5

;; CONST ENV SHAPES
#define CONST_ENV_SHAPE_1									1
#define CONST_ENV_SHAPE_2									2
#define CONST_ENV_SHAPE_3									3
#define CONST_ENV_SHAPE_4									4
#define CONST_ENV_SHAPE_5									5
#define CONST_ENV_SHAPE_6									6
#define CONST_ENV_SHAPE_7									7
#define CONST_ENV_SHAPE_8									8

;; BUTTONS CONST
;; -----------------------------------------------------------------------------------------------
#define CONST_BTN_ESC									0x00
#define CONST_BTN_SEL									0x01
#define CONST_BTN_ITEM_UP								0x02
#define CONST_BTN_ITEM_DWN								0x03
#define CONST_BTN_POS_LEFT								0x04
#define CONST_BTN_POS_RIGHT								0x05
#define CONST_BTN_VALUE_DEC								0x06
#define CONST_BTN_VALUE_INC								0x07
#define CONST_BTN_F1									0x08
#define CONST_BTN_F2									0x09
#define CONST_BTN_F3									0x0A
#define CONST_BTN_F4									0x0B
#define CONST_BTN_F5									0x0C
#define CONST_BTN_F6									0x0D
#define CONST_BTN_F7									0x0E
#define CONST_BTN_F8									0x0F
#define CONST_BTN_TONE_A								0x10
#define CONST_BTN_TONE_B								0x11
#define CONST_BTN_TONE_C								0x12
#define CONST_BTN_ADD_NOISE_A							0x13
#define CONST_BTN_ADD_NOISE_B							0x14
#define CONST_BTN_ADD_NOISE_C							0x15
#define CONST_BTN_ENVELOPE_A							0x16
#define CONST_BTN_ENVELOPE_B							0x17
#define CONST_BTN_ENVELOPE_C							0x18
#define CONST_BTN_FX_TAB_A								0x19
#define CONST_BTN_FX_TAB_B								0x1A
#define CONST_BTN_FX_TAB_C								0x1B


;; JUSTAGE FOR THE AUTOREPEAT BTN FUNCTION @ [+]/[-]
;; MORE INFORMATION @ user_tick_btn_auto_repeat.inc
;; 0 = Fast / 255 = Slow
#define CONST_BTN_AUTO_REPEAT_RATE						0x5F	;; ADJUST general AUTO BTN REPEAT RATE HERE
#define CONST_FIRST_BTN_STRIKE_CYCLE_TIME				0x8F	;; ADJUST the "first key strike" delay time here
#define CONST_KEY_AUTO_REPEAT_TIME						0x0F	;; ADJUST the AUTO REPEAT TIME here
#define CONST_BIT0_BTN_PERMANENT_PRESSED				0		;; user_tick_btn_auto_repeat.inc
																;; @ VAR_BTN_SPECIAL_FLAGS
																;; BIT0 = SET 	= BTN IS PERMANENT PRESSED = TRUE
																;; BIT0 = CLEAR	= BTN IS PERMANENT PRESSED = FALSE

;; PATCH - EEPROM & BANKSTICK
;; -----------------------------------------------------------------------------------------------
#define CONST_INTERNAL_EEPROM_PATCH_0					0		;; PATCH = INTERNAL PIC EEPROM PATCH
#define CONST_SELECT_CONF_DEFAULT_BANKSTICK				0		;; Default BS=BS1, BS1=0, BS2=1, BS3=2, ..., BS8=7
#define CONST_PIC_INT_EEPROM_FINGERPRINT_VALUE			0x12	;; Fingerprint Patch PIC EEPROM (Fingerprint = Formatted)
#define CONST_BANKSTICK_PATCH_FINGERPRINT_VALUE			0x88	;; Fingerprint Patch BANKSTICK (Fingerprint = Formatted)
#define CONST_ADR_PIC_EEPROM_PATCH_FINGERPRINT			0x00	;; ADRESS FINGERPRINT @ INT PATCH (EEPROM)
#define CONST_ADR_LOW_BYTE_PIC_EEPROM_ASC_NAME			0x01	;; POINTER TO THE PATCHNAME (FIRST ASC VALUE IN PIC EEPROM)
#define CONST_ADR_LOW_BYTE_BANKSTICKS_ASC_NAME			0x02	;; POINTER TO THE PATCHNAME (FIRST ASC VALUE IN BS)
#define CONST_ADR_LOW_BYTE_PIC_EEPROM_PATCH_DATA		0x09	;; LOW BYTE ADR - POINTER TO PIC EEPROM PATCH 0 DATA
#define CONST_ADR_LOW_BYTE_BANKSTICK_PATCH_DATA			0x0A	;; LOW BYTE ADR - POINTER TO BS PATCH (1-128) DATA
#define CONST_BANKSTICK_MAX_PATCH_NUMBER				127		;; 128 PATCHES: PATCH 0 in PIC EEPROM, PATCH 1-127 in BS
#define CONST_SET_BS_LOW_BYTE_ADR_COUNTER_TO_2			2		;; SEE @ patch_write_subroutines.inc
#define CONST_DEFAULT_PATCH_NUMBER_AFTER_BS_FORMATTING	1		;; BS PATCH 1 - SEE @ format_bankstick.inc

;; -----------------------------------------------------------------------------------------------
;; ==> BNKD_VAR_PATCH_STATE_FLAGS	
;; -----------------------------------------------------------------------------------------------
;; BIT0 = 1 = BANKSTICK	FOUND
;; BIT0 = 0 = NO BANKSTICK
;; BIT1 = 1 = CURRENT PATCH FINGERPRINT FOUND
;; BIT1 = 0 = CURRENT PATCH FINGERPRINT NOT FOUND
;; BIT2 = 1 = CURRENT PATCH = BANKSTICK    (PATCH 1-128)
;; BIT2 = 0 = CURRENT PATCH = PIC INTERNAL (PATCH 0)
;; BIT3 = 1 = DO FORMAT BS MODE = ON
;; BIT3 = 0 = DO FORMAT BS MODE = OFF
;; BIT4 = 1 = CURRENT PATCH NR =  BROWSE PATCH NR
;;	    = 1 = A PATCH HAD BEEN WRITTEN
;;	    = 1 = A PATCH HAD BEEN LOADED
;; BIT4 = 0 = CURRENT PATCH NR < > BROWSE PATCH NR
;; BIT5 = 1 = PATCH NAME HAS BEEN EDITED BY USER
;; BIT5 = 0 = PATCH NAME HAS NOT BEEN EDITED BY USER
;; BIT6 = 1 = PATCH NAME EDITING MODE = ON (USER CAN CHANGE THE PATCH NAME)
;; BIT6 = 0 = PATCH NAME EDITING MODE = OFF
;; BIT7 = 1 = A MIDI PROGRAM CHANGE EVENT HAS TAKEN PLACE Patch Load in Progress (Patch Change Request Flag)
;; BIT7 = 0 = NO PROGRAM CHANGE EVENT / THE PROGRAM CHANGE EVENT HAS BEEN CLEARD (NO Patch Change Request)
#define BIT0_BANKSTICK_FOUND							0
#define BIT1_FINGERPRINT_FOUND          				1
#define BIT1_CHECK_PATCH_FINGERPRINT       				1
#define BIT2_IF_SET_BS_PATCH_IS_SELECTED				2
#define BIT2_IF_CLR_PIC_EEPROM_PATCH_IS_SELECTED		2
#define BIT2_CHK_IF_PIC_EEPROM_PATCH_OR_BS_PATCH		2
#define BIT3_FORMAT_BANKSTICK_MODE_ON					3
#define BIT4_PATCH_HAS_BEEN_LOADED_OR_WRITTEN			4
#define BIT5_PATCH_NAME_WAS_EDITED_BY_USER				5
#define BIT6_PATCH_NAME_EDITING_MODE					6
#define BIT7_MIDI_PROGRAMM_CHANGE_RECEIVED				7

;; -----------------------------------------------------------------------------------------------
;;          Patch - CONST DEF
;; -----------------------------------------------------------------------------------------------
#define CONST_PATCH_NUMBER_VALUE_MAX							0x7F
#define CONST_PATCH_NUMBER_VALUE_IS_1							0x01
#define CONST_PATCH_NAME_CURSOR_POS_MAX							0x07
#define CONST_PATCH_NAME_ASC_MIN_VALUE							0x30	;; DEC 48 = ASC = "0"
#define CONST_PATCH_NAME_ASC_JMP_TO_ASC_A_VALUE					0x3A	;; DEC 58 = ASC = ":"
#define CONST_PATCH_NAME_ASC_JMP_TO_ASC_9_VALUE					0x40	;; DEC 64 = ASC = "@"
#define CONST_PATCH_NAME_ASC_9_VALUE							0x39	;; DEC 57 = ASC = "9"
#define CONST_PATCH_NAME_ASC_A_VALUE							0x41	;; DEC 65 = ASC = "A"
#define CONST_PATCH_NAME_ASC_MAX_VALUE_Z						0x5A	;; DEC 90 = ASC = "Z"


;; VELOCITY BIT @ TONE ABC | VELOCITY @ NOISE 
;; -----------------------------------------------------------------------------------------------
;; MEMO
;; VELOCITY 		  => NOISE PERIOD:
;; VELOCITY 	      => NOISE PERID is OFF	==> AY_VELOCITY_TO_ABC_NOISE_PERIOD_ON_OFF: BIT3=0, BIT5=0,BIT6=0,BIT7=0
;; VELOCITY of TONE A => NOISE PERIOD 		==> AY_VELOCITY_TO_ABC_NOISE_PERIOD_ON_OFF: BIT3=1, BIT5=1,BIT6=0,BIT7=0 
;; VELOCITY of TONE B => NOISE PERIOD 		==> AY_VELOCITY_TO_ABC_NOISE_PERIOD_ON_OFF: BIT3=1, BIT5=0,BIT6=1,BIT7=0 
;; VELOCITY of TONE C => NOISE PERIOD 		==> AY_VELOCITY_TO_ABC_NOISE_PERIOD_ON_OFF: BIT3=1, BIT5=0,BIT6=0,BIT7=1 
;; -----------------------------------------------------------------------------------------------
#define BIT0_VELOCITY_ON_OFF_TONE_A 								0
#define BIT1_VELOCITY_ON_OFF_TONE_B 								1
#define BIT2_VELOCITY_ON_OFF_TONE_C 								2
#define BIT3_VELOCITY_TO_NOISE_PERIOD_ON_OFF 						3
#define BIT5_VELOCITY_A_TO_NOISE_PERIOD_ON_OFF						5
#define BIT6_VELOCITY_B_TO_NOISE_PERIOD_ON_OFF						6
#define BIT7_VELOCITY_C_TO_NOISE_PERIOD_ON_OFF						7
#define CONST_VALUE32_FOR_NOISE_PERIOD_REVERSE_VALUE				32

;; VELOCITY BIT @ NOISE ABC | VELOCITY @ NOISE 
;; -----------------------------------------------------------------------------------------------
#define BIT5_NOISE_PERIOD_CONTROLED_VIA_VELOCITY_TONE_A_ON_OFF 		5
#define BIT6_NOISE_PERIOD_CONTROLED_VIA_VELOCITY_TONE_B_ON_OFF 		6
#define BIT7_NOISE_PERIOD_CONTROLED_VIA_VELOCITY_TONE_C_ON_OFF 		7

;; ENVELOPE MODE ON / OFF (BIT4 @ TONE ABC)
;; -----------------------------------------------------------------------------------------------
#define BIT4_ENVELOPE_ON_OFF										4
#define CONST_ENVELOPE_SHAPE_MAX_VALUE_8							8

;; FX ON/OFF @ TONE ABC
;; -----------------------------------------------------------------------------------------------
#define BIT0_FX_ON_OFF_TONE_A										0
#define BIT1_FX_ON_OFF_TONE_B										1
#define BIT2_FX_ON_OFF_TONE_C										2


;; USER DETUNE INI
#define CONST_INI_TUNE_FINE_USER									0x80	; INITIAL VALUE = 128 (0 on LCD!)
#define CONST_INI_TUNE_COARSE_USER									0x07 	; INITIAL VALUE = 7   (0 on LCD!)

;; DETUNE ABC
;; ----------
#define PROOF_POSITIVE_OR_NEGATIVE_VIA_0X06							0x06
#define PROOF_POSITIVE_OR_NEGATIVE_VIA_0X7F 						0x7F
#define OFFSET_COARSE_TUNE_IS_7										0x07
#define OFFSET_FINE_TUNE_IS_128 									0x80
#define PLUS_1_CARRY_BIT_KILLER										0x01

;; TUNE @ TONE ABC
;; MIN / MAX VALUE
;; ---------------
;; LOWEST TUNE - MAX VALUE
#define CONST_COARSE_TUNE_MAX_VALUE_B00001111						0x0F ;; dec 15,  hex 0x0F, B00001111
#define CONST_COARSE_TUNE_MAX_MINUS_1_B00001110						0x0E ;; dec 14,  hex 0x0E, B00001110
#define CONST_FINE_TUNE_MAX_VALUE_B11010010							0xD2 ;; dec 210, hex 0xD2, B11010010
;; HIGHEST TUNE - MIN VALUE
#define CONST_COARSE_TUNE_MIN_VALUE_B00000000						0x00 ;; dec 0,  hex 0x00, B00000000
#define CONST_FINE_TUNE_MIN_VALUE_B00001010							0x0A ;; dec 10, hex 0x0A, B00001010
;; MAX VALUE - NOISE PERIOD
#define NOISE_PERIOD_MAX_VALUE										0x1F ;; dec 31; bin 0001 1111
#define NOISE_PERIOD_MAX_VALUE_MINUS_1								0x1E
;; MAX VOLUME TONE ABC
#define VOLUME_ABC_MAX_VALUE										0x0F ;; dec 15; bin 0000 1111


;; SET NIBBLE
;; -----------------------------------------------------------------------------------------------
#define CONST_NIBBLE_B00001111										0x0F ;; dec 15, hex 0x0F, B00001111 


;; MIDIBox AY CS - LEDs
;; see @ dout_calls.inc
;; -----------------------------------------------------------------------------------------------
#define LED_F1											0x00			;; DOUT 00
#define LED_F2											0x01			;; DOUT 01
#define LED_F3											0x02			;; DOUT 02
#define LED_F4											0x03			;; DOUT 03
#define LED_F5											0x04			;; DOUT 04
#define LED_F6											0x05			;; DOUT 05
#define LED_F7											0x06			;; DOUT 06
#define LED_F8											0x07			;; DOUT 07
#define LED_ToneA										0x08			;; DOUT 08
#define LED_ToneB										0x09			;; DOUT 09
#define LED_ToneC										0x0A			;; DOUT 10
#define LED_NoiseA										0x0B			;; DOUT 11
#define LED_NoiseB										0x0C			;; DOUT 12
#define LED_NoiseC										0x0D			;; DOUT 13
#define LED_EnvelopeModeA								0x0E			;; DOUT 14
#define LED_EnvelopeModeB								0x0F			;; DOUT 15
#define LED_EnvelopeModeC								0x10			;; DOUT 16
#define LED_Fx_A										0x11			;; DOUT 17
#define LED_Fx_B										0x12			;; DOUT 18
#define LED_Fx_C										0x13			;; DOUT 19
#define LED_Envelope_Shape_1							0x14			;; DOUT 20
#define LED_Envelope_Shape_2							0x15			;; DOUT 21
#define LED_Envelope_Shape_3							0x16			;; DOUT 22
#define LED_Envelope_Shape_4							0x17			;; DOUT 23
#define LED_Envelope_Shape_5							0x18			;; DOUT 24
#define LED_Envelope_Shape_6							0x19			;; DOUT 25
#define LED_Envelope_Shape_7							0x1A			;; DOUT 26
#define LED_Envelope_Shape_8							0x1B			;; DOUT 27
#define LED_MIDI_CH_TONE_A_ACTIVITY						0x1C			;; DOUT 28 (since MBAY V2.3)
#define LED_MIDI_CH_TONE_B_ACTIVITY						0x1D			;; DOUT 29 (since MBAY V2.3)
#define LED_MIDI_CH_TONE_C_ACTIVITY						0x1E			;; DOUT 30 (since MBAY V2.3)
;; ---[PIN FOR HW-RESET THE AY-CHIP
#define HW_RESET_AY_CHIP								0x1F			;; DOUT 31 (last 4xDOUT PIN, (since MBAY V2.2)

;; LCD
#define PLUS_POS_Fx										5

;; MIDI / USER
;; -----------------------------------------------------------------------------------------------
#define INI_MIOS_MIDI_MERGER_ON_OFF						0x01	;; [ENABLED] | (1 = ENABLED, 0 = DISABLED)
;; MIDI CH int. [0-15]
;; MIDI CH ext. [1-16]
#define MIDI_CH_PLUS_1_OFFSETT							0x01
#define INI_TONE_A_MIDI_CHANNEL_USER					0x00 	; INITIAL VALUE = 0 = MIDI CH. 1 on LCD
#define INI_TONE_B_MIDI_CHANNEL_USER					0x01 	; INITIAL VALUE = 1 = MIDI CH. 2 on LCD
#define INI_TONE_C_MIDI_CHANNEL_USER					0x02 	; INITIAL VALUE = 2 = MIDI CH. 3 on LCD
;; MIDI SWITCH ABC TONE/NOISE ON OF
#define	INI_MIDI_ALL_OFF 								0x3F 	; INITIAL VALUE = BIN 00 111 111
																; AY REG 07     =     IO CBA CBA
																; 	    		=     BA
																; 	    		=        NOISE
																; 	    		=            TONE
																; ABC = 1 = TONE OFF
																
#define INI_USER_TONE_ABC_NOISE_ABC_ON_OFF				0xF8	;; USER SETTING
																;; HEX 0xF8
																;; BIN 11 111 000
																;;	  CBA CBA
																;;	  NOI TONE
																
#define INI_VELOCITY_TO_ACB_NOISE_PERIOD				0x00 	;; VELOCITY TO ABC VOLUME = OFF
																;; NOTICE:
																;; VELOCITY TO NOISE PERIOD = OFF
																;; BIN 0000 0000 
																;;          NCBA = ON VELOCITY @ NOISE & TONE ABC=0 =off 
																;;
																;; VELOCITY TO NOISE PERIOD = ON
																;; BIN 0000 1111 
																;;          NCBA = ON VELOCITY @ NOISE & TONE ABC=1 =on 
																;; NOISE:
																;; NOISE  ON (BIT3 SET) VELOCITY
																;; NOISE  OFF(BIT3 CLR) VELOCITY
																;; [FOR DETAILS SEE abowe: => "VELOCITY BIT TONE / NOISE ABC"]
																
#define INI_FX_USER_ON_OFF_ABC							0x00	;; After Startup: All Fx = OFF
																;; USER: FX ABC off (See MAIN MENU 3 / Mixer)
																;; BIT0 = A (SET = ON / CLR = OFF)
																;; BIT1 = B (SET = ON / CLR = OFF)
																;; BIT2 = C (SET = ON / CLR = OFF)
																
#define CONST_MIDI_FILTER_ALL_OFF_BIN_00111111			0x3F	; BIN 00111111 (Do not change)


							
#define CONST_FX_FUNC0									0x00	;"Fx STOP"
#define CONST_FX_FUNC1									0x01	; "ON  A"
#define CONST_FX_FUNC2									0x02	; "OFF A"
#define CONST_FX_FUNC3									0x03	; "ON  B"
#define CONST_FX_FUNC4									0x04	; "OFF B"
#define CONST_FX_FUNC5									0x05	; "ON  C"
#define CONST_FX_FUNC6									0x06	; "OFF C"
#define CONST_FX_FUNC7									0x07	; "TuneA+PARAM"
#define CONST_FX_FUNC8									0x08	; "TuneA-PARAM"
#define CONST_FX_FUNC9									0x09	; "TuneB+PARAM"
#define CONST_FX_FUNC10									0x0A	; "TuneB-PARAM"
#define CONST_FX_FUNC11									0x0B	; "TuneC+PARAM"
#define CONST_FX_FUNC12									0x0C	; "TuneC-PARAM"
#define CONST_FX_FUNC13									0x0D	; "Vol.A PARAM"
#define CONST_FX_FUNC14									0x0E	; "Vol.B PARAM"
#define CONST_FX_FUNC15									0x0F	; "Vol.C PARAM"
#define CONST_FX_FUNC16									0x10	; "Noise ON  A"
#define CONST_FX_FUNC17									0x11	; "Noise OFF A"
#define CONST_FX_FUNC18									0x12	; "Noise ON  B"
#define CONST_FX_FUNC19									0x13	; "Noise OFF B"
#define CONST_FX_FUNC20									0x14	; "Noise ON  C"
#define CONST_FX_FUNC21									0x15	; "Noise OFF C"
#define CONST_FX_FUNC22									0x16	; "N.Frq.PARAM"
#define CONST_FX_FUNC23									0x17	; "Env. ON  A" 
#define CONST_FX_FUNC24									0x18	; "Env. OFF A"
#define CONST_FX_FUNC25									0x19	; "Env. ON  B" 
#define CONST_FX_FUNC26									0x1A	; "Env. OFF B"
#define CONST_FX_FUNC27									0x1B	; "Env. ON  C" 
#define CONST_FX_FUNC28									0x1C	; "Env. OFF C"
#define CONST_FX_FUNC29									0x1D	; "EnvShapePRM"
#define CONST_FX_FUNC30									0x1E	; "EnvTimeFine"
#define CONST_FX_FUNC31									0x1F	; "EnvTimeCrse"
#define CONST_FX_FUNC32									0x20	; "Note#1 >> A"
#define CONST_FX_FUNC33									0x21	; "Note#1 >> B"
#define CONST_FX_FUNC34									0x22	; "Note#1 >> C"
#define CONST_FX_FUNC35									0x23	; "Note#2 >> A"
#define CONST_FX_FUNC36									0x24	; "Note#2 >> B"
#define CONST_FX_FUNC37									0x25	; "Note#2 >> C"
#define CONST_FX_FUNC38									0x26	; "Note#3 >> A"
#define CONST_FX_FUNC39									0x27	; "Note#3 >> B"
#define CONST_FX_FUNC40									0x28	; "Note#3 >> C"
#define CONST_FX_FUNC41									0x29	; "GoPos:PARAM"	


;; CC MAX (MIDI Controller 1-29)
#define CC_MAX_VALUE_29									0x1D	;; 0x1D = 29

;; MIDI CC
#define CC22_VALUE_1									0x01
#define CC22_VALUE_2									0x02
#define CC22_VALUE_3									0x03

;; MIDI PITCH BEND
#define CC_30_PITCH_BEND								0x1E	;; 30 dec
#define CHK_POS_NEG_VIA_63								0x3F	;; 64-1 (63 dec)
#define CLEAR_OFFSET_128								0x80	;; 128

