#ifndef _FILE_H
#define _FILE_H

#ifdef __cplusplus
extern "C" {
#endif

/////////////////////////////////////////////////////////////////////////////
// Global definitions
/////////////////////////////////////////////////////////////////////////////

// error codes
// NOTE: FILE_SendErrorMessage() should be extended whenever new codes have been added!

#define FILE_ERR_SD_CARD           -1 // failed to access SD card
#define FILE_ERR_NO_PARTITION      -2 // DFS_GetPtnStart failed to find partition
#define FILE_ERR_NO_VOLUME         -3 // DFS_GetVolInfo failed to find volume information
#define FILE_ERR_UNKNOWN_FS        -4 // unknown filesystem (only FAT12/16/32 supported)
#define FILE_ERR_OPEN_READ         -5 // DFS_OpenFile(..DFS_READ..) failed, e.g. file not found
#define FILE_ERR_OPEN_READ_WITHOUT_CLOSE -6 // FILE_ReadOpen() has been called while previous file hasn't been closed via FILE_ReadClose()
#define FILE_ERR_READ              -7 // DFS_ReadFile failed
#define FILE_ERR_READCOUNT         -8 // less bytes read than expected
#define FILE_ERR_READCLOSE         -9 // DFS_ReadClose aborted due to previous error
#define FILE_ERR_WRITE_MALLOC     -10 // FILE_WriteOpen failed to allocate memory for write buffer
#define FILE_ERR_OPEN_WRITE       -11 // DFS_OpenFile(..DFS_WRITE..) failed
#define FILE_ERR_OPEN_WRITE_WITHOUT_CLOSE -12 // FILE_WriteOpen() has been called while previous file hasn't been closed via FILE_WriteClose()
#define FILE_ERR_WRITE            -13 // DFS_WriteFile failed
#define FILE_ERR_WRITECOUNT       -14 // less bytes written than expected
#define FILE_ERR_WRITECLOSE       -15 // DFS_WriteClose aborted due to previous error
#define FILE_ERR_SEEK             -16 // FILE_Seek() failed
#define FILE_ERR_OPEN_DIR         -17 // DFS_OpenDir(..DFS_READ..) failed, e.g. directory not found
#define FILE_ERR_COPY             -18 // FILE_Copy() failed
#define FILE_ERR_COPY_NO_FILE     -19 // source file doesn't exist
#define FILE_ERR_NO_DIR           -20 // FILE_GetDirs() or FILE_GetFiles() failed because of missing directory
#define FILE_ERR_NO_FILE          -21 // FILE_GetFiles() failed because of missing directory
#define FILE_ERR_SYSEX_READ       -22 // error while reading .syx file
#define FILE_ERR_MKDIR            -23 // FILE_MakeDir() failed
#define FILE_ERR_INVALID_SESSION_NAME -24 // FILE_LoadSessionName()
#define FILE_ERR_UPDATE_FREE      -25 // FILE_UpdateFreeBytes()
#define FILE_ERR_REMOVE           -26 // FILE_Remove() failed


/////////////////////////////////////////////////////////////////////////////
// Global Types
/////////////////////////////////////////////////////////////////////////////

// simplified file reference, part of FIL structure of FatFs
typedef struct {
  u8  flag;  // file status flag
  u8  csect; // sector address in cluster
  u32 fptr;  // file r/w pointer
  u32 fsize; // file size
  u32 org_clust; // file start cluster
  u32 curr_clust; // current cluster
  u32 dsect; // current data sector;
  u32 dir_sect; // sector containing the directory entry
  u8 *dir_ptr; // pointer to the directory entry in the window
} file_t;


/////////////////////////////////////////////////////////////////////////////
// Prototypes
/////////////////////////////////////////////////////////////////////////////

extern s32 FILE_Init();

extern s32 FILE_CheckSDCard(void);

extern s32 FILE_SDCardAvailable(void);
extern s32 FILE_VolumeAvailable(void);
extern u32 FILE_VolumeBytesFree(void);
extern u32 FILE_VolumeBytesTotal(void);
extern char *FILE_VolumeLabel(void);
extern s32 FILE_UpdateFreeBytes(void);

extern u32 FILE_VolumeSectorsPerCluster(void);
extern u32 FILE_VolumeCluster2Sector(u32 cluster);

extern s32 FILE_ReadOpen(char *filepath);
extern s32 FILE_ReadClose();
extern s32 FILE_ReadSeek(u32 offset);
extern u32 FILE_ReadGetCurrentSize(void);
extern u32 FILE_ReadGetCurrentPosition(void);
extern s32 FILE_ReadBuffer(u8 *buffer, u32 len);
extern s32 FILE_ReadBufferUnknownLen(u8 *buffer, u32 len);
extern s32 FILE_ReadLine(u8 *buffer, u32 max_len);
extern s32 FILE_ReadByte(u8 *byte);
extern s32 FILE_ReadHWord(u16 *hword);
extern s32 FILE_ReadWord(u32 *word);

extern s32 FILE_WriteOpen(char *filepath);
extern s32 FILE_WriteClose(void);
extern s32 FILE_WriteSeek(u32 offset);
extern u32 FILE_WriteGetCurrentSize(void);
extern u32 FILE_WriteGetCurrentPosition(void);
extern s32 FILE_WriteBuffer(u8 *buffer, u32 len);
extern s32 FILE_WriteByte(u8 byte);
extern s32 FILE_WriteHWord(u16 hword);
extern s32 FILE_WriteWord(u32 word);

extern s32 FILE_Copy(char *src_file, char *dst_file);

extern s32 FILE_MakeDir(char *path);

extern s32 FILE_Remove(char *path);

extern s32 FILE_FileExists(char *filepath);
extern s32 FILE_DirExists(char *path);

extern s32 FILE_GetDirs(char *path, char *dir_list, u8 num_of_items, u8 dir_offset);
extern s32 FILE_GetFiles(char *path, char *ext_filter, char *file_list, u8 num_of_items, u8 file_offset);



/////////////////////////////////////////////////////////////////////////////
// Export global variables
/////////////////////////////////////////////////////////////////////////////

// last error status returned by DFS
// can be used as additional debugging help if FILE_*ERR returned by function
extern u32 file_dfs_errno;

// for percentage display during copy operations
extern u8 file_copy_percentage;

#ifdef __cplusplus
}
#endif

#endif /* _FILE_H */
