// COIL WINDER
//  by Michael Sigl in 2023
//  noncommercial use
// 	based on the MIOS32 Platform > www.ucapps.de > by Thorsten Klose


#include <mios32.h>
#include "app.h"
#include <FreeRTOS.h>
#include <portmacro.h>
#include <task.h>
#include <queue.h>
#include <semphr.h>
#include "tasks.h"
#include "file.h"
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <wchar.h> // for copy pattern-memory

#include "app_lcd.h"	// "local" because stripped down to SSD1306 only

#include "glcd_font.h"

// TASKS
static void LCD(void *pvParameters);
static void SD(void *pvParameters);
static void MOTOR(void *pvParameters);

// TASK   - PRIORITYS
#define PRIORITY_SD				( tskIDLE_PRIORITY + 3 ) // idle
#define PRIORITY_MOTOR			( tskIDLE_PRIORITY + 3 ) // idle
#define PRIORITY_LCD			( tskIDLE_PRIORITY + 0 ) // idle


static u8 ENCset(u8 doit);

xSemaphoreHandle xLCDSemaphore;
#define MUTEX_LCD_TAKE { while( xSemaphoreTakeRecursive(xLCDSemaphore, (portTickType)1) != pdTRUE ); }
#define MUTEX_LCD_GIVE { xSemaphoreGiveRecursive(xLCDSemaphore); }

xSemaphoreHandle xSDCardSemaphore; 	// take and give access to SD-Card



/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

// STEPPER MOTOR VARIABLES
//
// Endswitches
u8 EndL = 0;
u8 EndR = 0;

// Start the Winding Program
u8 start = 0;	// 0 means it is not started...

// Displaying the Program Status
int steps = 0;	// Turner - how many Steps
int turns = 0;	// Turner - how many Turns

// Actual Feeder Distance in [mm] - in relation to the home position
float actual_distance = 0;


// Define pi as a constant
const float pi = 3.141592653589793238462643383279502884197f;


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// SD-Card /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
u8 file_type[4] 	= "0000";    // definie the Filetype which is later saved via "FILE_WriteBuffer" into the file header
char path[8] 	= "A/0.t4  ";
u16 Save   = 1000;
u16 Load   = 1000;

char copyfilepath[8] = {};

file_t midifile_fi;		// used to read files from SD-Card

u16 NrLoad  = 0;
u16 NrStor  = 0;
u16 main_song = 0;

s8 Menue = 0;
u8 updLCD = 1;

s8  Cursor_V = 2;

// in order to display what the winder acutal does...
u8 windingstage = 0;

u8 running = 0;
int counter = 0;

// Winding Program Parameters
    typedef struct windprogram {   // to optimize SD-Load-Time, each Parameter is stored in this struct

        char	name[8];
        int    	turns;
        int		speed;
        int		length_cable;
        int		dia_zylinder;
        int		length_zylinder;
        float	wire;
        float	space;
        int		start_distance;
        int		end_distance;
        int		start_offset;
        int		zeropoint;
        int		direction;
        int		painttime;
    } windprogram_t;

    windprogram_t wind;



// Float to Integer Conversion  STRUCT
typedef struct {
				uint16_t whole;
				uint16_t fraction;
} FloatToIntResult;



/// ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



void APP_Init(void){


	// Winding Programm initial Values
	wind.turns = 200;
	wind.speed = 1;
	wind.length_cable = 300;
	wind.length_zylinder = 200;
	wind.dia_zylinder = 40;
	wind.wire = 0.3;
	wind.space = 0.2;
	wind.start_distance = 20;
	wind.end_distance = 20;
	wind.zeropoint = 20;
	wind.start_offset = 45;
	wind.direction = 0;
	wind.painttime = 1;	//in minutes



	u16 c,x = 0;

	// initialize LEDs
	MIOS32_BOARD_LED_Init(0xffffffff);

	// LCD  ///////////////////////////////////////////
		MIOS32_LCD_DeviceSet(0);    MIOS32_LCD_Init(0);
		APP_LCD_Clear();
		xLCDSemaphore  = xSemaphoreCreateRecursiveMutex();			// create MUTEX


/// SD - Card //////////// CHECK /////////////////////////////////////////////////////////////////////////////////////////////

	s8 statusSD = 0;			// Status of the SDCard--connected? fat?

	xSDCardSemaphore  = xSemaphoreCreateRecursiveMutex();
	MIOS32_SDCARD_Init(0);
	FILE_Init(0);
	xTaskCreate(SD,  "SD", 	512, NULL, PRIORITY_SD, NULL);

			// SD-Card  is  it there?
			statusSD = MIOS32_SDCARD_CheckAvailable(0);
			statusSD = FILE_CheckSDCard();

				// Print to LCD - that there is a CARD
				MUTEX_LCD_TAKE;
					MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);						// use default font
					MIOS32_LCD_DeviceSet( 0 ); // MENUE LCD
						MIOS32_LCD_CursorSet(0, 0);	MIOS32_LCD_PrintFormattedString("SD-CARD  check/init  ");
						MIOS32_LCD_CursorSet(0, 1);	MIOS32_LCD_PrintFormattedString("                     ");
						MIOS32_LCD_CursorSet(0, 2);
						if (statusSD == 0) 	{   	MIOS32_LCD_PrintFormattedString("No CARD-restart it!  ");}
						if (statusSD) 	{ 			MIOS32_LCD_PrintFormattedString("CARD ok              ");}
						MIOS32_LCD_CursorSet(0, 3);	MIOS32_LCD_PrintFormattedString("                     ");
						MIOS32_LCD_CursorSet(0, 4);	MIOS32_LCD_PrintFormattedString("                     ");
						MIOS32_LCD_CursorSet(0, 5);	MIOS32_LCD_PrintFormattedString("                     ");
						MIOS32_LCD_CursorSet(0, 6);	MIOS32_LCD_PrintFormattedString("                     ");
						MIOS32_LCD_CursorSet(0, 7);	MIOS32_LCD_PrintFormattedString("                     ");
				MUTEX_LCD_GIVE;
/// /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////



/// SD - CARD - Make Directory and Files//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		if (statusSD > 0)	{	// only if there is a Card!

						s8 FileState = 0;		// Status "is" your Patch there?


						///////////// D I R   -   PATCH ///////////////////////////////////////////////////////////

								s8 statusDir = 0;			// Status "is" your direction "sdcard/a/" there?

								// reduce Directory "A/0.sy"  to  "A    "
								strncpy(path,  "A       ", 8);

								MUTEX_SDCARD_TAKE;
										// Directory /a/?
										statusDir 	= FILE_DirExists(path);	 		// ask file.c: exist a folder "A/" on the CARD? //

										// No Directory make one...
										if (statusDir <= 0){	FILE_MakeDir(path);

																// Check if Dir now Exists
																statusDir 	= FILE_DirExists(path);	} // ask file.c: exist a folder "A/" on the CARD? //
								MUTEX_SDCARD_GIVE;

								// LCD
								MUTEX_LCD_TAKE;
										MIOS32_LCD_DeviceSet( 0 );
										MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);
										MIOS32_LCD_CursorSet(0, 3);
										if (statusDir <= 0) 	{ MIOS32_LCD_PrintFormattedString("Directory A/ FAIL");}
										if (statusDir >  0) 	{ MIOS32_LCD_PrintFormattedString("DIR ok          ");}
								MUTEX_LCD_GIVE;



						///////////// F I L E S    -   PATCH
								char copyfilepath[8];

								// Build File Path/0.t4      "
								sprintf   (path, "A/127.t4");

								// Check Files
								MUTEX_SDCARD_TAKE;
										FileState  =  FILE_FileExists(path);	// ask file.c: exist a file "a/255.t4" on the CARD? //
								MUTEX_SDCARD_GIVE;


								// LCD
								MUTEX_LCD_TAKE;
										MIOS32_LCD_DeviceSet( 0 );
										if( FileState <= 0) 	{ MIOS32_LCD_CursorSet(0, 4);  MIOS32_LCD_PrintFormattedString("127.t4 not found      ");}
										if( FileState == 1) 	{ MIOS32_LCD_CursorSet(0, 4);  MIOS32_LCD_PrintFormattedString("127.t4 found          ");}
								MUTEX_LCD_GIVE;


								// Create File - If not existent
								if ( FileState <= 0 && statusDir) {

																		// Build File Path/0.t4      "
																		sprintf   (path, "A/0.t4");

																		MUTEX_SDCARD_TAKE;
																					// Write Data into file
																					FILE_WriteOpen	( path,       8);
																						FILE_WriteBuffer( (u8 *)file_type,	4); //"SIGL"     = 4 Positons
																						FILE_WriteBuffer( (u8 *)&wind,       	sizeof(windprogram_t) );
																					FILE_WriteClose	();

																					FileState = FILE_FileExists(path);	// ask file.c: exist a file "A/0.t4" on the CARD?
																		MUTEX_SDCARD_GIVE;


																		if(FileState) {	// Patch- copy multiple times

																								// LCD     prepare the user that he have to wait forever!
																								MUTEX_LCD_TAKE;
																									MIOS32_LCD_DeviceSet( 0 );
																									MIOS32_LCD_CursorSet(0, 5);  MIOS32_LCD_PrintFormattedString("file 0.t4 created     ");
																								    MIOS32_LCD_CursorSet(0, 7);  MIOS32_LCD_PrintFormattedString("it counts to 127!     ");
																								MUTEX_LCD_GIVE;

																								for(c=1; c<128; c++) {	sprintf (copyfilepath, "A/%d.t4",c);  // make a new filename depending on the counter value

																														// LCD
																														MUTEX_LCD_TAKE;
																															MIOS32_LCD_DeviceSet( 0 );                              //  ("                      ")
																															MIOS32_LCD_CursorSet(0, 6);  MIOS32_LCD_PrintFormattedString("0.t4 copy > %d.t4 ",c);
																														MUTEX_LCD_GIVE;

																														// Copy FILEs
																														MUTEX_SDCARD_TAKE;
																															FILE_Copy ( (char *)path, (char *)copyfilepath ); // copy the File 0.t4 to all other 256 files...
																														MUTEX_SDCARD_GIVE;
																														}
																								}

																		// Check filesystem
																		MUTEX_SDCARD_TAKE;
																				FileState  =  FILE_FileExists("A/127.t4");
																		MUTEX_SDCARD_GIVE;


																		// Print to LCD - that there is a File
																		MUTEX_LCD_TAKE;
																				MIOS32_LCD_DeviceSet( 0 );
																					if (FileState <= 0) { MIOS32_LCD_CursorSet(0, 7);  MIOS32_LCD_PrintFormattedString("copy FAILT!           ");}
																					if (FileState) 		{ MIOS32_LCD_CursorSet(0, 7);  MIOS32_LCD_PrintFormattedString("127.t4 found          ");}
																		MUTEX_LCD_GIVE;
																	}


						///////////// F I L E S    -   Previos Set

								// Check Files
								MUTEX_SDCARD_TAKE;
										FileState = FILE_FileExists("0.p4");	// ask file.c: exist a file "0.p4" on the CARD? //
								MUTEX_SDCARD_GIVE;


								// LCD
								MUTEX_LCD_TAKE;
										MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);
										MIOS32_LCD_DeviceSet( 0 );
										if( FileState <= 0)	{ MIOS32_LCD_CursorSet(0, 4);  MIOS32_LCD_PrintFormattedString("0.p4   not found     ");}
										if( FileState )		{ MIOS32_LCD_CursorSet(0, 4);  MIOS32_LCD_PrintFormattedString("previos set found    ");}
								MUTEX_LCD_GIVE;


								// Load Previos Set (if existent)
								if( FileState ) {

															MUTEX_SDCARD_TAKE;
																	// Read Data from file
																	FILE_ReadOpen	( &midifile_fi, "0.p4");
																			FILE_ReadBuffer( (u8 *)file_type,  4); //"SIGL" = 4 Positons
							                                                FILE_ReadBuffer( (u8 *)&NrStor, sizeof(NrStor) );
							                                        FILE_ReadClose	(&midifile_fi);

																	FileState  =  FILE_FileExists("0.p4");
															MUTEX_SDCARD_GIVE;

															// LCD
															MUTEX_LCD_TAKE;
																	MIOS32_LCD_DeviceSet( 0 );													  // "                      ");}
																	if( FileState )	{ 	MIOS32_LCD_CursorSet(0, 5);  MIOS32_LCD_PrintFormattedString("PreSet: %d  ", main_song);
																						}
															MUTEX_LCD_GIVE;


															// Load it!
															NrLoad 	= NrStor;
															Load   	= main_song;

															//MIOS32_MIDI_SendDebugMessage("load lastbpm: %d: main_song: %d", lastBPM, main_song);
														}

								// Create FileStructure
								if( FileState <= 0 && statusDir) {
																		MUTEX_SDCARD_TAKE;
																				FILE_WriteOpen	("0.p4", 8);
																						FILE_WriteBuffer( (u8 *)file_type,	4);
										                                                FILE_WriteBuffer( (u8 *)&NrStor, sizeof(NrStor)  );
										                                        FILE_WriteClose	();

																				FileState  =  FILE_FileExists("0.p4");	// ask file.c: exist a file "A/0.p4" on the CARD? //
																		MUTEX_SDCARD_GIVE;

																		// LCD
																		MUTEX_LCD_TAKE;
																						MIOS32_LCD_DeviceSet( 0 );
																						if( FileState ) {	MIOS32_LCD_CursorSet(0, 5);  MIOS32_LCD_PrintFormattedString("file 0.p4 created    ");}
																		MUTEX_LCD_GIVE;
																		}
			} // Endo of YES CARD



	// LCD Task
	xTaskCreate(LCD, "LCD", configMINIMAL_STACK_SIZE, NULL, PRIORITY_LCD, NULL);

	// Lo Priority Task
	xTaskCreate(MOTOR, "MOTOR", configMINIMAL_STACK_SIZE, NULL, PRIORITY_MOTOR, NULL);


	const mios32_enc_config_t encoders[4] = {//(SR begin with 1, ENC with 0) // setup the Pinout of Encoders  // The BLM16x16+x has 5x 165er Shiftregister chips...
											{ .cfg.type=DETENTED1, .cfg.speed=SLOW, 	.cfg.speed_par=1, .cfg.sr=1, .cfg.pos=0 },      //	0  PAGE
											{ .cfg.type=DETENTED2, .cfg.speed=NORMAL, 	.cfg.speed_par=1, .cfg.sr=1, .cfg.pos=2 },      //	1  PAR 1
											{ .cfg.type=DETENTED2, .cfg.speed=NORMAL, 	.cfg.speed_par=1, .cfg.sr=1, .cfg.pos=4 },      //	1  PAR 1
											{ .cfg.type=DETENTED2, .cfg.speed=NORMAL, 	.cfg.speed_par=3, .cfg.sr=1, .cfg.pos=6 }};		//	2  PAR 2
											u8 i;	for(i=0; i<4; ++i) { MIOS32_ENC_ConfigSet(i, encoders[i]);	 }

	ENCset(0);	// Initialize Encoder Speeds

	Menue = 0; 	// LCD Menue Page = depending of MtxNr



	// Motor Controll Pins
	MIOS32_BOARD_J10_PinInit(0, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Feeder	Pulse
	MIOS32_BOARD_J10_PinInit(1, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Feeder	Direction
	MIOS32_BOARD_J10_PinInit(2, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Fedder	Enable
	MIOS32_BOARD_J10_PinInit(3, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Turner	Pulse
	MIOS32_BOARD_J10_PinInit(4, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Turner	Direction
	MIOS32_BOARD_J10_PinInit(5, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);	//	Turner	Enable

	// Motor Pins Init Value
	for(x=0;x<6;x++)	{ MIOS32_BOARD_J10_PinSet(x, 0); }

	// Motor Driver  -  NPN - Disable
	MIOS32_BOARD_J10_PinSet(2,1);	// Feeder
	MIOS32_BOARD_J10_PinSet(5,1);	// Turner

	// Update LCDs
	updLCD = 1;

}



void APP_Background(void){}



void APP_Tick(void){ // @ mS rate, from main task which also handles DIN, ENC and AIN events.  dont do more than 300uS!


	// Isolation Painting - Stage - minutes Counter
	if(running && windingstage==6){
									// each Secund counter
									static int mscounter = 0;
									mscounter++;
									if(mscounter >= 999){	mscounter = 0;
															counter++;
														}

									if(counter >= (wind.painttime*60)){	counter = 0;
																		windingstage = 7;	// aka finished
																		updLCD = 1;
																		running = 0;
																		MIOS32_BOARD_J10_PinSet(5,1);	// Turner	De-Activate
																		}
									}
}



void APP_MIDI_NotifyPackage( mios32_midi_port_t port, mios32_midi_package_t midi_package ){} // End Hook



void APP_SRIO_ServicePrepare(void)  {}



void APP_SRIO_ServiceFinish(void)   {

	static u8 onetimer = 1;

	if(onetimer) {	onetimer = 0;
					// Get END-Switch Positions
					EndL = !MIOS32_DIN_PinGet( 22 );
					EndR = !MIOS32_DIN_PinGet( 23 );

						MUTEX_LCD_TAKE;
						MIOS32_LCD_DeviceSet( 1 );
							MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
							MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("        ");	// clear
							MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("EndL: %d",EndL);
							MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
							MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("EndR %d",EndR);
					MUTEX_LCD_GIVE;
				}

}



void APP_DIN_NotifyToggle(u32 pin, u32 pin_value){

	u8 val = !pin_value;

	//MIOS32_MIDI_SendDebugMessage("pin %d   value%d", pin, val);

	//////////////////////////////////////////
	// U I   PINOUT							//
	//										//
	//	ENCbtn						ENCbtn	//
	//	8							11		//
	//			ENCbtn		ENCbtn			//
	//			9			10				//
	//////////////////////////////////////////


	switch(pin){	case 22:	EndL = val;	break;	// Endswitch Left
					case 23:	EndR = val;	break;	// Endswitch Right
					case 12:	if(val){
										if(windingstage==0) {	running = 1;		// Not running... then run!
																windingstage = 1; 	// start first Stage "go to homeposition"
																steps = 0;			// reset the Turn Counter at startup
																turns = 0;
																updLCD = 1;
																}
										if(windingstage==1) { running =! running;	break; }	//
										if(windingstage==2) { running =! running;	break; }	//
										if(windingstage==3) { running =! running;	break; }	//
										if(windingstage==4) { running =! running;	break; }	//
										if(windingstage==5) { running =! running;	break; }	//
										if(windingstage==6) { running =! running;	break; }	//
										if(windingstage==7) { running =! running;	break; }	//

										}



					case 9:		if(Menue==0){ Load = NrLoad; break;}

					case 10:	if(Menue==0){ Save = NrStor; break;}

					break; }

}



void APP_ENC_NotifyChange(u32 encoder, s32 incrementer){

	s8 inc = incrementer;	// for shorter code
	s16	z = 0;
	u8 enc = encoder;

	switch(Menue)	{	// Store Load ////////////////////////////////////////////////////////////////////////////
						case 0:
									// Menue Select
									if (enc == 0){
													z = Menue + inc;		// get Parameter-value
													if( z >= 4 ) { z = 4; }	// clip
													if( z <= 0 ) { z = 0; }	// clip
													Menue = z;				// set new Parameter-value

													updLCD = 1;	// update LCD
													ENCset(0);
													break;
													}

									// Load Select
									if (enc == 1){
													z = NrLoad + inc;
													if( z >= 127 )	{ z = 127; }
													if( z <= 0 ) 	{ z = 0; }
													NrLoad = z;

													updLCD = 1;
													break;
													}


									// Store Select
									if (enc == 2){
													z = NrStor + inc;
													if( z >= 127 )	{ z = 127; }
													if( z <= 0 ) 	{ z = 0; }
													NrStor = z;

													updLCD = 1;
													break;
													}

										// not in use
									if (enc == 3){
													updLCD = 1;
													break;
													}
									break;



						// Edit Preset /////////////////////////////////////////////////////////////////////
						case 1:
									// Menue Select
									if (enc == 0){
													z = Menue + inc;		// change Menue Page
													if( z >= 4 ) { z = 4; }	// clip
													if( z <= 0 ) { z = 0; }	// clip
													Menue = z;				// set the Menue-Page

													updLCD = 1;	// update LCD
													ENCset(0);
													break;
													}

									// Cursor
									if (enc == 1){

													z = Cursor_V + inc;
													if (z < 0)	{ z = 0;	}
													if (z > 8)	{ z = 8;	}
													Cursor_V = z;

													updLCD = 1;	// update LCD
													break;
													}


									// not in use
									if (enc == 2){


													updLCD = 1;	// update LCD
													break;
													}

									// Edit Parameter Value
									if (enc == 3){
													if(Cursor_V==0) {

																		z = wind.turns + inc;
																		if (z < 1)	{ z = 1;	}
																		if (z > 2000)	{ z = 2000;	}
																		wind.turns = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==1) {
																		float f = wind.wire;
																		if(inc < 0) { f = f - 0.01; }
																		if(inc > 0) { f = f + 0.01; }

																		if (f < 0.01){ f = 0.01;	}
																		if (f > 2)	 { f = 2;	}
																		wind.wire = f;

																		updLCD = 1;
																		break;}

													if(Cursor_V==2) {
																		float f = wind.space;
																		if(inc < 0) { f = f - 0.01; }
																		if(inc > 0) { f = f + 0.01; }

																		if (f < 0.01){ f = 0.01;	}
																		if (f > 20)	 { f = 20;	}
																		wind.space = f;

																		updLCD = 1;
																		break;}

													if(Cursor_V==3) {
																		z = wind.painttime + inc;
																		if (z < 0)	{ z = 0;	}
																		if (z > 240){ z = 240;	}
																		wind.painttime  = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==4) {
																		z = wind.start_offset + inc;
																		if (z < 1)	{ z = 1;	}
																		if (z > 400){ z = 400;	}
																		wind.start_offset  = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==5) {
																		z = wind.start_distance + inc;
																		if (z < 1)	{ z = 1;	}
																		if (z > 400){ z = 400;	}
																		wind.start_distance  = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==6) {
																		z = wind.end_distance + inc;
																		if (z < 1)	{ z = 1;	}
																		if (z > 400){ z = 400;	}
																		wind.end_distance  = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==7) {
																		z = wind.direction + inc;
																		if (z < 0){ z = 0;	}
																		if (z > 1){ z = 1;	}
																		wind.direction  = z;

																		updLCD = 1;
																		break;}

													if(Cursor_V==8) {
																		z = wind.dia_zylinder + inc;
																		if (z < 0){ z = 0;	}
																		if (z > 300){ z = 300;	}
																		wind.dia_zylinder  = z;

																		updLCD = 1;
																		break;}




													break;
													}
									break;



						case 2:
									// Menue Select
									if (enc == 0){
													z = Menue + inc;		// get Parameter-value
													if( z >= 4 ) { z = 4; }	// clip
													if( z <= 0 ) { z = 0; }	// clip
													Menue = z;				// set new Parameter-value

													updLCD = 1;	// update LCD
													ENCset(0);
													break;
													}

									break;



						case 3:
									// Menue Select
									if (enc == 0){
													z = Menue + inc;		// get Parameter-value
													if( z >= 4 ) { z = 4; }	// clip
													if( z <= 0 ) { z = 0; }	// clip
													Menue = z;				// set new Parameter-value

													updLCD = 1;	// update LCD
													ENCset(0);
													break;
													}

									break;



						case 4:
									// Menue Select
									if (enc == 0){
													z = Menue + inc;		// get Parameter-value
													if( z >= 4 ) { z = 4; }	// clip
													if( z <= 0 ) { z = 0; }	// clip
													Menue = z;				// set new Parameter-value

													updLCD = 1;	// update LCD
													ENCset(0);
													break;
													}

									break;




							break;
						}



}


static u8 ENCset(u8 doit) {	// change the Encoders Settings ON THE FLY!
return 0;
}



// SD - C A R D  ///////////////////////////////////////////////////////////////////////////
static void SD(void *pvParameters) {

  portTickType xLastExecutionTime;


  // Initialise the xLastExecutionTime variable on task entry
  xLastExecutionTime = xTaskGetTickCount();

  while( 1 ) {
				vTaskDelayUntil(&xLastExecutionTime, 1 / portTICK_RATE_MS);


// S T O R E /////////////////////////////////////////////////////////////////////////////////////////////////////

		// Triggermatrix + Song.Struct
		if( Save != 1000 )	{
									// calc Filepath		   //256 songs   4Seq-Variations
									sprintf(path, "A/%d.t4", Save);

									MUTEX_SDCARD_TAKE;
											// Write Data into file
											FILE_WriteOpen	( path,       8);
												FILE_WriteBuffer( (u8 *)file_type,	4); //"SIGL"     = 4 Positons
												FILE_WriteBuffer( (u8 *)&wind,       	sizeof(windprogram_t) );
											FILE_WriteClose	();
									MUTEX_SDCARD_GIVE;

									NrLoad = NrStor;


									// Save it as "Previos Set" in order to load it on next boot...
									MUTEX_SDCARD_TAKE;
											// Write Data into file
											FILE_WriteOpen	("0.p4", 8);
													FILE_WriteBuffer( (u8 *)file_type,	4); //"SIGL" = 4 Positons
	                                                FILE_WriteBuffer( (u8 *)&NrStor, sizeof(NrStor)  );
	                                        FILE_WriteClose	();
									MUTEX_SDCARD_GIVE;

									updLCD = 1;		// Update Displays

									Save = 1000; // Kill Flag
					}


// L O A D /////////////////////////////////////////////////////////////////////////////////////////////////////


		// Triggermatrix + Song.struct
		if( Load != 1000 )	{
									// calc Filepath		 // 256 songs
									sprintf(path, "A/%d.t4", Load);

									MUTEX_SDCARD_TAKE;
											// Read Data from file
											FILE_ReadOpen	( &midifile_fi, path);
													FILE_ReadBuffer( (u8 *)file_type,	4); //"SIGL"     = 4 Positons
													FILE_ReadBuffer( (u8 *)&wind,       	sizeof(windprogram_t) );
											FILE_ReadClose	( &midifile_fi );
									MUTEX_SDCARD_GIVE;

									updLCD = 1;		// Update Displays

									Load = 1000;	// kill Flag
							}
}
}




// Motorcontroll
static void MOTOR(void *pvParameters) {

  portTickType xLastExecutionTime;


  // Initialise the xLastExecutionTime variable on task entry
  xLastExecutionTime = xTaskGetTickCount();

  while( 1 ) {	vTaskDelayUntil(&xLastExecutionTime, 1 );


				// Break the sequence
				if(running){




							// DM542 Pinout /////////////////////////////////////////////
							//
							//	JP10 Pin	DM542	Feeder
							//	0			PULse
							//	1			DIRection
							//	2			ENAble
							//
							//	JP10 Pin	DM542	Turner
							//	3			PULse
							//	4			DIRection
							//	5			ENAble


							// TURNER /////////////////////////////////////////////
							// the Turner Motor is a "422HD4027-37 :
							//	*	1,8° per Step
							//	*	360° / 1,8° = 200 Steps/360 turn
							//
							// a DM542 Driver set to Microstep "400" needs 800 steps for a 360° turn
							//
							// because we toggle the state to move it... its 800 not 400





							// FEEDER ////////////////////////////////////////////////////////////////
							static float travel = 0.005;	// mm/step
							static float endtravel = 0;
							static float normtravel = 0;

							// FAIL SAVE

							// Feeder is most-left (switch is activated)
							static u8 direction = 0;		// 0 = Left  1 = Right
							if(EndL) { direction = 1; }		// >>	change direction to right

							// Feeder is most-right (switch is activated)
							if(EndR)  { direction = 0; }	// <<	change direction to left

							// Set direction
							MIOS32_BOARD_J10_PinSet(1,direction);








							// Find home
							if(windingstage==1) {

													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,0);	// Feeder	// 1: Deactivate, 0: Activate
													MIOS32_BOARD_J10_PinSet(5,1);	// Turner

													// Direction	-	fail save...
													static u8 direction = 0;	// <<	0 = Left  1 = Right

													// Feeder is most-left (switch is activated)
													if(EndL) { 	direction = 1; 			// >>	change direction to right
																actual_distance = 0;	// we are on zeropoint
																windingstage = 2;		// show next stage on LCD
																updLCD = 1;
																}

													// Feeder is most-right (switch is activated)
													if(EndR)  { direction = 0; }		// <<	change direction to LEFT




													// Set direction
													MIOS32_BOARD_J10_PinSet(1,direction);

													// Move the Feeder
													static u8 move = 0;
													move =! move;	// toggle State (Pulse)
													MIOS32_BOARD_J10_PinSet(0,move);
													}


							// Go to start position ///////////////////////////////////////////////////////////////////////////////
							if(windingstage==2) {

													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,0);	// Feeder
													MIOS32_BOARD_J10_PinSet(5,1);	// Turner	Deactivate


															// FEEDER //////////////////////////////////////////////


															// Direction	-	fail save...//////////////////////

															// Feeder is most-left (switch is activated)
															if(EndL) { direction = 1; }		// >>	change direction to right

															// Feeder is most-right (switch is activated)
															if(EndR)  { direction = 0; }	// <<	change direction to left

															// Set direction
															MIOS32_BOARD_J10_PinSet(1,direction);




															// lets set a travel-distance
															actual_distance = actual_distance + travel;

															// toggle Pulse State
															static u8 move = 0;
															move =! move;

															// Send Pulses to the Driver
															MIOS32_BOARD_J10_PinSet(0,move);

															if( actual_distance >= wind.start_offset) {
																											windingstage = 3;	// indicater next Stage on LCD
																											updLCD = 1;
																											running = 0;		// break the automatic sequence, and wait for a button press
																										}
															}



							// Isolation Winding (Start)	// at full speed
							if(windingstage==3){

													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,0);	// Feeder
													MIOS32_BOARD_J10_PinSet(5,0);	// Turner


													// TURNER ////////////////////////////////////////////
													steps++;
													if(steps >= 800) {	steps = 0;
																		turns++;
																		updLCD = 1;	// want to show the actual number off Turns
																		}

																		static u8 moove = 0;
																		moove =! moove;	// toggle state
																		MIOS32_BOARD_J10_PinSet(3,moove);	// Moove


																		MIOS32_BOARD_J10_PinSet(4,wind.direction);	// Direction

													// FEEDER ////////////////////////////////////////////

													// lets set a travel-distance
													actual_distance = actual_distance + travel;

													// toggle Pulse State
													static u8 move = 0;
													move =! move;

													// Send Pulses to the Driver
													MIOS32_BOARD_J10_PinSet(0,move);

													// end the sequence
													if( actual_distance >= (wind.start_distance + wind.start_offset) ) {	normtravel = 0;
																															windingstage = 4;
																															updLCD = 1;
																														}
													}




							// Isolation Winding (END)	// at full speed
							if(windingstage==5){

													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,0);	// Feeder
													MIOS32_BOARD_J10_PinSet(5,0);	// Turner


													// TURNER ////////////////////////////////////////////////////////////////
													steps++;
													if(steps >= 800) {	steps = 0;
																		turns++;	}

													static u8 moove = 0;
													moove =! moove;	// toggle state
													MIOS32_BOARD_J10_PinSet(3,moove);	// Moove


													MIOS32_BOARD_J10_PinSet(4,wind.direction);	// Direction




													// FEEDER ////////////////////////////////////////////////////////////////

													// lets set a travel-distance
													endtravel = endtravel + travel;

													// toggle Pulse State
													static u8 move = 0;
													move =! move;

													// Send Pulses to the Driver
													MIOS32_BOARD_J10_PinSet(0,move);

													// end ALL
													if( endtravel >= wind.end_distance ) {	start = 0;
																							windingstage = 6;
																							updLCD = 1;
																							running = 0;	// break sequence --- and wait for button press
																						}
													}




							// normal Winding
							if(windingstage==4){

													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,0);	// Feeder
													MIOS32_BOARD_J10_PinSet(5,0);	// Turner


													// TURNER ////////////////////////////////////////////
													steps++;
													if(steps >= 800){	steps = 0;
																		turns++;
																		updLCD = 1;	// want to show the actual number off Turns

																		if(turns >= wind.turns){	endtravel = 0;
																									windingstage = 5;	// go to endisolation state
																									updLCD = 1;
																									}
																		}

																		static u8 moove = 0;
																		moove =! moove;	// toggle state
																		MIOS32_BOARD_J10_PinSet(3,moove);	// Move

																		MIOS32_BOARD_J10_PinSet(4,wind.direction);	// Direction


													// FEEDER ////////////////////////////////////////////
													static float travel = 0.005;	// mm/step


													// some math /////////////////////////////////////////////////////
													float travel_per_turn = wind.wire + wind.space;	// 0.9 = 0.5 + 0.4

													travel_per_turn = travel_per_turn / 4; 		// 0.225 = 0.9 / 4

													float stepping = 800.0 * travel_per_turn; 	// 180 = 800 * 0.225
													int stpr = 800 / stepping;						// just to make a integer 	because: steppa = steps % stpr; dont work with FLOAT



													static int steppa = 0;

													// Remember the 0-Step
													u16 oldvalue = steppa;


													//	>> so each 180 Steps off in a 800 step loop we moove the Feeder

													steppa = steps % stpr; // counting from 0-12 in a endless loop


													// MIOS32_MIDI_SendDebugMessage("steppa %d = steps %d  /  stpr %d", steppa, steps, stpr);


													if( (steppa != oldvalue) && (!steppa) ) {
																								// toggle Pulse State
																								static u8 move = 0;
																								move =! move;

																								// Send Pulses to the Driver
																								MIOS32_BOARD_J10_PinSet(0,move);

																								// Remember Actual Distance
																								normtravel = normtravel + travel;
																								}
													}



							// Isolation Painting Stage
							if(windingstage==6){


													// Motor Driver  -  NPN - Enable and Disable
													MIOS32_BOARD_J10_PinSet(2,1);	// Feeder	Deactivate
													MIOS32_BOARD_J10_PinSet(5,0);	// Turner	Activate


													// TURNER ////////////////////////////////////////////
													steps++;
													if(steps >= 800) {	steps = 0;
																		updLCD = 1;	// want to show the actual number off Minutes
																		}

																		static u8 moove = 0;
																		moove =! moove;	// toggle state
																		MIOS32_BOARD_J10_PinSet(3,moove);	// Moove

																		MIOS32_BOARD_J10_PinSet(4,wind.direction);	// Direction
												}
							}
			}
}


// Float to INT   Converter
FloatToIntResult FloatToInt(float f) {
									    FloatToIntResult result;
									    result.whole = (uint16_t)f;
									    result.fraction = (uint16_t)(f * 100) % 100;
									    return result;
}



// Calculate the cross-sectional area of a wire  ( Querschnitt)
static inline float WireArea(float diameter) {

											    const float radius = diameter / 2000.0f;
											    return pi * radius * radius;
}

// Calculate the resistance of a wire	( Copper)
static inline float WireResistance(float length) {

											    const float area = WireArea(wind.wire);
											    const float resistivity = 1.68e-8f;
											    return resistivity * length / area;
}







static void LCD(void *pvParameters) {

  portTickType xLastExecutionTime;
  xLastExecutionTime = xTaskGetTickCount();

  while(1) {

			vTaskDelayUntil(&xLastExecutionTime, 100);

			if(updLCD){	// Kill Flag
						updLCD = 0;

						switch(Menue){
										// Load - Store Menue //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////7
										case 0:


												// Screen 1:	Laden
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 0 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 2);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("LOAD %d", NrLoad);

														MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);			// ("                     ");
														MIOS32_LCD_CursorSet(0, 3);	MIOS32_LCD_PrintFormattedString("Windings: %d", wind.turns);
														MIOS32_LCD_CursorSet(0, 4);	MIOS32_LCD_PrintFormattedString("Pipe-L:%d  DN:%d", wind.length_cable, wind.dia_zylinder);

														// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
														FloatToIntResult result = FloatToInt(wind.wire);
														int Wire_whole = result.whole;
														int Wire_fraction = result.fraction;
														result = FloatToInt(wind.space);

														MIOS32_LCD_CursorSet(0, 5);
														MIOS32_LCD_PrintFormattedString("Coil:%d.%02d Space:%d.%02d", Wire_whole, Wire_fraction, result.whole, result.fraction);

														MIOS32_LCD_CursorSet(0, 7);	MIOS32_LCD_PrintFormattedString("Enc-Press = LOAD ");
												MUTEX_LCD_GIVE;



												// Screen 2:	Speichern
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 1 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 2);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("SAVE %d", NrStor);

														MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);			// ("                     ");
														MIOS32_LCD_CursorSet(0, 3);	MIOS32_LCD_PrintFormattedString("Windings: %d", wind.turns);
														MIOS32_LCD_CursorSet(0, 4);	MIOS32_LCD_PrintFormattedString("Pipe-L:%d  DN:%d", wind.length_cable, wind.dia_zylinder);

														// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
														result = FloatToInt(wind.wire);
														Wire_whole = result.whole;
														Wire_fraction = result.fraction;
														result = FloatToInt(wind.space);

														MIOS32_LCD_CursorSet(0, 5);
														MIOS32_LCD_PrintFormattedString("Coil:%d.%02d Space:%d.%02d", Wire_whole, Wire_fraction, result.whole, result.fraction);


														MIOS32_LCD_CursorSet(0, 7);	MIOS32_LCD_PrintFormattedString("Enc-Press = SAVE");
												MUTEX_LCD_GIVE;
												break;




										// Edit Preset ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////7
										case 1:


												// Screen 1:	Select Parameter to edit
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 0 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("       1");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 2);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear

														MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);			// ("                     ");

														// Cursor	-- only if it is on this Screen --- it goes over 2 screens...
														if(Cursor_V <= 7){	MIOS32_LCD_CursorSet(0, Cursor_V);	MIOS32_LCD_PrintFormattedString(">"); }

														MIOS32_LCD_CursorSet(1, 0);	MIOS32_LCD_PrintFormattedString("Windings: %d", wind.turns);

														// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
														result = FloatToInt(wind.wire);
														MIOS32_LCD_CursorSet(1, 1);
														MIOS32_LCD_PrintFormattedString("Wire-Di: %d.%02d mm", result.whole, result.fraction);

														// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
														result = FloatToInt(wind.space);
														MIOS32_LCD_CursorSet(1, 2);
														MIOS32_LCD_PrintFormattedString("Wind-Space: %d.%02d mm", result.whole, result.fraction);

														MIOS32_LCD_CursorSet(1, 3);	MIOS32_LCD_PrintFormattedString("PaintTime: %d min",  wind.painttime);


														MIOS32_LCD_CursorSet(1,4);	MIOS32_LCD_PrintFormattedString("Start-Offset: %dmm",  wind.start_offset);
														MIOS32_LCD_CursorSet(1,5);	MIOS32_LCD_PrintFormattedString("StartIsoWind: %dmm",  wind.start_distance);
														MIOS32_LCD_CursorSet(1,6);	MIOS32_LCD_PrintFormattedString("End  IsoWind: %dmm",  wind.end_distance);

														if(wind.direction)	{ MIOS32_LCD_CursorSet(1, 7);	MIOS32_LCD_PrintFormattedString("WindTurn left"); }
														else 				{ MIOS32_LCD_CursorSet(1, 7);	MIOS32_LCD_PrintFormattedString("WindTurn rigt"); }

												MUTEX_LCD_GIVE;


												// Screen 2:	Edit Parameter Value   &&  and see some math
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 1 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 2);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear


														// Show Parameter to Edit in BIG
														MIOS32_LCD_CursorSet(0, 1);									   //   "        " 	// use big custom font
														if(Cursor_V==0) {	MIOS32_LCD_PrintFormattedString("Wind:%d", wind.turns); }

														if(Cursor_V==1) {	// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
																		    FloatToIntResult result = FloatToInt(wind.wire);
																		    MIOS32_LCD_PrintFormattedString("Di: %d.%02d", result.whole, result.fraction);
																		}


														if(Cursor_V==2) {	// Convert FLOAT to INTEGER --- because the Display cant handle FLOATS directly
																		    result = FloatToInt(wind.space);
																		    MIOS32_LCD_PrintFormattedString("Sp: %d.%02d", result.whole, result.fraction);
																		}

														if(Cursor_V==3) {	MIOS32_LCD_PrintFormattedString("Ptme: %d", wind.painttime); }

														if(Cursor_V==4) {	MIOS32_LCD_PrintFormattedString("Soff: %d", wind.start_offset); }

														if(Cursor_V==5) {	MIOS32_LCD_PrintFormattedString("Siso: %d", wind.start_distance); }

														if(Cursor_V==6) {	MIOS32_LCD_PrintFormattedString("Eiso: %d", wind.end_distance); }
																															  //  "        "
														if(Cursor_V==7) {	if(wind.direction)	{ MIOS32_LCD_PrintFormattedString("TurnLEFT"); }
																			else 				{ MIOS32_LCD_PrintFormattedString("TurnRigt"); }
																		}

														if(Cursor_V==8) {	MIOS32_LCD_PrintFormattedString("DiaP:%d", wind.dia_zylinder); }



														// Math // Länge des Rohres
														float geschlaenge = (wind.turns * (wind.wire+wind.space)) + wind.start_distance + wind.end_distance;
														int laenge = geschlaenge;

														MIOS32_LCD_FontInit((u8 *)GLCD_FONT_NORMAL);
														// Cursor	-- only if it is on this Screen --- it goes over 2 screens...
														if(Cursor_V >=8){	MIOS32_LCD_CursorSet(0, (Cursor_V-8));	MIOS32_LCD_PrintFormattedString(">"); }


														MIOS32_LCD_CursorSet(1, 0);	MIOS32_LCD_PrintFormattedString("Pipe Diam: %d mm",  wind.dia_zylinder);


														MIOS32_LCD_CursorSet(0, 4);						// ("                     ");

														if(Cursor_V==0) {	MIOS32_LCD_PrintFormattedString("incl. Start/End Winds");}

														if(Cursor_V==1) {	MIOS32_LCD_PrintFormattedString("Diameter off Wire    ");}

														if(Cursor_V==2) {	MIOS32_LCD_PrintFormattedString("2get more Viso btween");}

														if(Cursor_V==3) {	MIOS32_LCD_PrintFormattedString("Minutes to Dry-Turn  ");}

														if(Cursor_V==4) {	MIOS32_LCD_PrintFormattedString("start Position Pipe  ");}

														if(Cursor_V==5) {	MIOS32_LCD_PrintFormattedString("Good: Pipe-Diam / 2 ");}

														if(Cursor_V==6) {	MIOS32_LCD_PrintFormattedString("Good: Pipe-Diam / 2 ");}

														if(Cursor_V==7) {	MIOS32_LCD_PrintFormattedString("Coil Wind  Direction");}

														if(Cursor_V==8) {	MIOS32_LCD_PrintFormattedString("Diameter off Pipe   ");}


														// Cable-Length
														float length = (wind.turns * (wind.dia_zylinder*pi)) / 1000.0;
														MIOS32_LCD_CursorSet(0, 6);	MIOS32_LCD_PrintFormattedString("Cable Length: %d m", (int)length);

														// Resistance
														MIOS32_LCD_CursorSet(0, 5);	MIOS32_LCD_PrintFormattedString("Resistance: %d Ohm", (int)WireResistance(length));

														MIOS32_LCD_CursorSet(0, 7);	MIOS32_LCD_PrintFormattedString("Coil  Length: %dmm", laenge);
												MUTEX_LCD_GIVE;
												break;




										// Coil Winding Process ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////7
										case 2:


												// Screen 1:
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 0 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("Winding ");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear
														MIOS32_LCD_CursorSet(0, 2);  MIOS32_LCD_PrintFormattedString("        ");	// clear

														// winding Stage
														MIOS32_LCD_CursorSet(0, 2);
														if(windingstage==6) {	MIOS32_LCD_PrintFormattedString("Min: %d", counter/60); }
														else { 					MIOS32_LCD_PrintFormattedString("Trn:%d", turns); }
																												//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear

												MUTEX_LCD_GIVE;


												// Screen 1:
												MUTEX_LCD_TAKE;
													MIOS32_LCD_DeviceSet( 1 );
														MIOS32_LCD_FontInit((u8 *)fnt_BIG);   					//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 0);  MIOS32_LCD_PrintFormattedString("Stage:  ");	// clear
														MIOS32_LCD_CursorSet(0, 1);  MIOS32_LCD_PrintFormattedString("        ");	// clear

														// winding Stage
														MIOS32_LCD_CursorSet(0, 2);
														if(windingstage==0) {  MIOS32_LCD_PrintFormattedString("Needhome");	}
														if(windingstage==1) {  MIOS32_LCD_PrintFormattedString("FindHome");	}
														if(windingstage==2) {  MIOS32_LCD_PrintFormattedString("StartOff");	}
														if(windingstage==3) {  MIOS32_LCD_PrintFormattedString("StartIso");	}
														if(windingstage==4) {  MIOS32_LCD_PrintFormattedString("NormWind");	}
														if(windingstage==5) {  MIOS32_LCD_PrintFormattedString("EndIso  ");	}
														if(windingstage==6) {  MIOS32_LCD_PrintFormattedString("IsoPaint");	}
														if(windingstage==7) {  MIOS32_LCD_PrintFormattedString("finished");	}
																												//   "        " 	// use big custom font
														MIOS32_LCD_CursorSet(0, 3);  MIOS32_LCD_PrintFormattedString("        ");	// clear

												MUTEX_LCD_GIVE;
												break;



										break;


							}
						}
				}
}
