// $Id: app.c 690 2009-08-04 22:49:33Z tk $
/*
 * MIOS32 Wavedrum MIDI
 * Version 1.0
 * ==========================================================================
 *
 *  Copyright (C) 2013 Ingo Debus (<your email address>)
 *  Licensed for personal non-commercial use only.
 *  All other rights reserved.
 * 
 * ==========================================================================
 */

/////////////////////////////////////////////////////////////////////////////
// Include files
/////////////////////////////////////////////////////////////////////////////

#include <mios32.h>
#include "app.h"
#include "patch.h"
#include "sysex.h"
#include "adc.h"
#include "timer.h"
#include "params.h"
#include "MIDI.h"


/////////////////////////////////////////////////////////////////////////////
// defines
/////////////////////////////////////////////////////////////////////////////

#define SEND_WAVE		//send adc buffer contents as sysex message

#define MIDI_PORT		DEFAULT	//UART0

#define SYX_LEN			(7+BUF_SIZE*5+1)
#define XTRIG_SUPPRESS	TRUE
#define PRES_SCALE		11		//adc output is divided by this value for CC and AT
#define PRES_SCALE_PB	4		//adc output is multiplied with this value for PB


/////////////////////////////////////////////////////////////////////////////
// global variables
/////////////////////////////////////////////////////////////////////////////

volatile u8 patch;
volatile u8 bank;

s32 zero_lvl[ADC_NUM];
#ifdef SENSE_PRESSURE
s32 zero_prs;
#endif
u16 l_thresh[ADC_NUM];
u16 h_thresh[ADC_NUM];


/////////////////////////////////////////////////////////////////////////////
// extern global variables
/////////////////////////////////////////////////////////////////////////////

extern vu8 trig_state[ADC_NUM];
extern vu32 buffer[ADC_NUM][BUF_SIZE];
extern vs32 adc_min[ADC_NUM];
extern vs32 adc_max[ADC_NUM];
extern vu8 abv_thr_cnt[ADC_NUM];	//count the values that are above threshold, to filter out glitches
extern vu8 blw_thr_cnt[ADC_NUM];	//count the values that are below threshold, to filter out glitches
#ifdef SENSE_PRESSURE
extern vu8 prs_state;
extern vu16 pressure;
#endif

/////////////////////////////////////////////////////////////////////////////
// Function prototypes
/////////////////////////////////////////////////////////////////////////////

u8 get_vel(u8 adc_chnl);
#ifdef SEND_WAVE
void send_wave(u8 adc_chnl);
#endif
void send_note(u8 adc_chnl);

s32 SYSEX_SendAck(mios32_midi_port_t port, u8 ack_code, u8 ack_arg);	//!!

/////////////////////////////////////////////////////////////////////////////
// Functions
/////////////////////////////////////////////////////////////////////////////

//get note velocity from maxima in buffer
//adc_chnl: 0=Rim; 1=Head
u8 get_vel(u8 adc_chnl)
{
	u8 vel;
	
	vel = (MAX(zero_lvl[adc_chnl]-adc_min[adc_chnl], adc_max[adc_chnl]-zero_lvl[adc_chnl])) / (PATCH_ReadByte(RIM_VEL_SCALE + adc_chnl) + 1);	//velocity
	
	if(vel > 127)
		vel = 127;
	return vel;
}

#ifdef SEND_WAVE
void send_wave(u8 adc_chnl)
{
	u8 i;
	
	PATCH_WriteWord(ZERO0, zero_lvl[0]);
	PATCH_WriteWord(ZERO1, zero_lvl[1]);

	//write detected min and max
	PATCH_WriteWord(DETMIN, adc_min[adc_chnl]-zero_lvl[adc_chnl]);
	PATCH_WriteWord(DETMAX, adc_max[adc_chnl]-zero_lvl[adc_chnl]);

	//write indicator which adc channel is the current one
	PATCH_WriteByte(ACTIVE_CHNL, adc_chnl);

	//write MSByte of waveform into patch buffer
	for(i=0; i<BUF_SIZE; i++)
		PATCH_WriteByte(WAVE0+i, buffer[adc_chnl][i] >> 4);
	
	SYSEX_Send(MIDI_PORT, 0, 0);
}
#endif

//adc_chnl: 0=Rim; 1=Head
void send_note(u8 adc_chnl)
{	
	u8 note, midichnl;
	
	note = PATCH_ReadByte(RIM_NOTE + adc_chnl) & 0x7f;
	midichnl = PATCH_ReadByte(RIM_MIDICH + adc_chnl) & 0x0f;
	
	MIOS32_MIDI_SendNoteOn(MIDI_PORT, midichnl, note, 0);
	MIOS32_MIDI_SendNoteOn(MIDI_PORT, midichnl, note, get_vel(adc_chnl));

#ifdef SEND_WAVE
	if(PATCH_ReadByte(SENDWAVE))
		send_wave(adc_chnl);
#endif
	
	adc_min[adc_chnl] = zero_lvl[adc_chnl];	//init min detector
	adc_max[adc_chnl] = zero_lvl[adc_chnl];	//init max detector
	abv_thr_cnt[adc_chnl] = 0;		//init above-threshold counter
	blw_thr_cnt[adc_chnl] = 0;		//init below-threshold counter
	trig_state[adc_chnl] = READY;
	
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called after startup to initialize the application
/////////////////////////////////////////////////////////////////////////////
void APP_Init(void)
{
	u8 i;

	MIOS32_MIDI_SendDebugMessage("start initializing...\n");

	// initialize all LEDs
	MIOS32_BOARD_LED_Init(0xffffffff);

	// initialise SysEx parser
	SYSEX_Init(0);
	
	// initialize patch structure
	PATCH_Init(0);
	
	// init local patch/bank
	patch = bank = 0;

	// initialize all J10 Pins
	for(i=0; i<7; i++)
		MIOS32_BOARD_J10_PinInit(i, MIOS32_BOARD_PIN_MODE_OUTPUT_PP);
	
	/* Initialize ADC  */
	ADCInit( ADC_CLK );
	
	/* Init Timer */
	init_timer0();
	enable_timer(0);

	SYSEX_SendAck(MIDI_PORT, 0, 0);

}


/////////////////////////////////////////////////////////////////////////////
// This task is running endless in background
/////////////////////////////////////////////////////////////////////////////
void APP_Background(void)
{
	u8 i;
	u8 adc_chnl;
	u8 chnl;
	u8 XTrigFlag;
#ifdef SENSE_PRESSURE
	s16 old_adc;
	s16 new_adc;
	
	old_adc = 0;
#endif
	XTrigFlag = RESET;
	
	//wait for a while until offset has stabilized
	for(i=0; i<100; i++)				//100 * 50ms = 5s
		MIOS32_DELAY_Wait_uS(50000);	//50ms delay
	
	//fill buffer0 once to get zero level
	MIOS32_MIDI_SendDebugMessage("measure zero Rim...\n");
	trig_state[0] = TRIGD;
	while (trig_state[0] != WAIT);		//wait until buffer is ready to be processed

	for(i=0, zero_lvl[0]=0; i<BUF_SIZE; i++)
		zero_lvl[0] += buffer[0][i];
	zero_lvl[0] /= BUF_SIZE;				//average buffer
	MIOS32_MIDI_SendDebugMessage("Zero Rim = %d\n", zero_lvl[0]);
	PATCH_WriteWord(ZERO0, zero_lvl[0]);
	
	//fill buffer1 once to get zero level
	MIOS32_MIDI_SendDebugMessage("measure zero Head...\n");
	trig_state[1] = TRIGD;
	while (trig_state[1] != WAIT);		//wait until buffer is ready to be processed

	for(i=0, zero_lvl[1]=0; i<BUF_SIZE; i++)
		zero_lvl[1] += buffer[1][i];
	zero_lvl[1] /= BUF_SIZE;				//average buffer
	MIOS32_MIDI_SendDebugMessage("Zero Head = %d\n", zero_lvl[1]);
	PATCH_WriteWord(ZERO1, zero_lvl[1]);
#ifdef SENSE_PRESSURE												//meanwhile output pressure data
	MIOS32_MIDI_SendDebugMessage("measure zero Pressure...\n");
	for(i=0, zero_prs=0; i<BUF_SIZE; i++)
	{
		while(prs_state == READY);		//wait until pres data is available
		zero_prs += pressure;
	}
	zero_prs /= BUF_SIZE;
	zero_prs -= 50;		//set the offset a tad lower than zero level, to suppress noise
	MIOS32_MIDI_SendDebugMessage("Zero Pressure = %d\n", zero_prs);
	PATCH_WriteWord(ZEROPRS, zero_prs);
#endif
	SYSEX_Send(MIDI_PORT, 0, 0);
	
	h_thresh[0] = zero_lvl[0] + PATCH_ReadByte(THRESH);
	l_thresh[0] = zero_lvl[0] - PATCH_ReadByte(THRESH);
	h_thresh[1] = zero_lvl[1] + PATCH_ReadByte(THRESH);
	l_thresh[1] = zero_lvl[1] - PATCH_ReadByte(THRESH);
	
	trig_state[0]  = trig_state[1]  = READY;
	abv_thr_cnt[0] = abv_thr_cnt[1] = 0;		//init above-threshold counter
	blw_thr_cnt[0] = blw_thr_cnt[1] = 0;		//init below-threshold counter
	adc_min[0]     = adc_max[0]     = zero_lvl[0];	//init min/max detector
	adc_min[1]     = adc_max[1]     = zero_lvl[1];	//init max/max detector
		
	// endless loop
	while( 1 ) {
		// toggle the state of LED (allows to measure the execution speed with a scope)
//		MIOS32_BOARD_LED_Set(1, ~MIOS32_BOARD_LED_Get());
		
		while ((trig_state[0] != WAIT) && (trig_state[1] != WAIT))	//wait until one of the buffers is ready to be processed
		{
#ifdef SENSE_PRESSURE												//meanwhile output pressure data
			if(PATCH_ReadByte(PRES_CC) <= CC_PBDN && prs_state == WAIT)		//PRES_CC isn't set to Off && new pres data is available
			{
				new_adc = zero_prs - pressure;
				prs_state = READY;
				if(PATCH_ReadByte(PRES_CC) <= CC_AT)	//PRES_CC is CC or Aftertouch, range 0..127
				{
					new_adc /= PRES_SCALE;
					if(new_adc < 0)
						new_adc = 0;
					if(new_adc > 127)
						new_adc = 127;
					if(((new_adc > 2) || (old_adc > 0)) && (new_adc != old_adc) && (new_adc < old_adc + 20))
					{
						if(PATCH_ReadByte(PRES_CC) < CC_AT)
							MIOS32_MIDI_SendCC(MIDI_PORT, PATCH_ReadByte(PRES_MIDICH), PATCH_ReadByte(PRES_CC), new_adc);
						else if(PATCH_ReadByte(PRES_CC) == CC_AT)
							MIOS32_MIDI_SendAftertouch(MIDI_PORT, PATCH_ReadByte(PRES_MIDICH), new_adc);
						old_adc = new_adc;
					}
				}
				else	//PRES_CC is Pitch bend up or down
				{
					new_adc *= PRES_SCALE_PB;
					if(new_adc < 0)
						new_adc = 0;
					if(new_adc > 8191)
						new_adc = 8191;
					if(((new_adc > 50) || (old_adc > 0)) && (new_adc != old_adc) /*&& (new_adc < old_adc + 20)*/)
					{
						MIOS32_MIDI_SendPitchBend(MIDI_PORT, PATCH_ReadByte(PRES_MIDICH), (PATCH_ReadByte(PRES_CC) == CC_PBUP) ? (8192 + new_adc) : (8192 - new_adc));
						old_adc = new_adc;
					}
				}
			}
#endif
		}
		for(adc_chnl=0; adc_chnl<ADC_NUM; adc_chnl++)
		{
			if(trig_state[adc_chnl] == WAIT)
			{
				h_thresh[adc_chnl] = zero_lvl[adc_chnl] + PATCH_ReadByte(THRESH);	//update threshold here, this means...
				l_thresh[adc_chnl] = zero_lvl[adc_chnl] - PATCH_ReadByte(THRESH);	//...it get updated AFTER the next hit...
				if(XTrigFlag)										//XTrigFlag is set, there's a pending note from the other piezo
				{
					XTrigFlag = RESET;
					if(abv_thr_cnt[adc_chnl] > PATCH_ReadByte(ABV_THR_CNT_MAX))		//no glitch, output current or pending note, whichever has higher vel
						chnl = get_vel(adc_chnl) > get_vel(~adc_chnl&0x01) ? adc_chnl : ~adc_chnl&0x01;
					else											//glitch, output pending note
						chnl = ~adc_chnl&0x01;
					send_note(chnl);								//send current or pending note
					//reset the other note
					adc_min[~chnl&0x01] = adc_max[~chnl&0x01] = zero_lvl[~chnl&0x01];					//init min/max detector
					abv_thr_cnt[~chnl&0x01] = 0;					//init above-threshold counter
					blw_thr_cnt[~chnl&0x01] = 0;					//init below-threshold counter
					trig_state[~chnl&0x01] = READY;
				}
				else												//XTrigFlag is not set
				{
					if(abv_thr_cnt[adc_chnl] > PATCH_ReadByte(ABV_THR_CNT_MAX))		//no glitch, do note output
					{
						if(!XTRIG_SUPPRESS || trig_state[~adc_chnl&0x01] == WAIT)		//nothing going on on other piezo, so send note right now
							send_note(adc_chnl);
						else										//other piezo is active too, so just set flag
							XTrigFlag = SET;
					}
					else											//glitch, ignore
					{
						adc_min[adc_chnl] = adc_max[adc_chnl] = zero_lvl[adc_chnl];	//init min/max detector
						abv_thr_cnt[adc_chnl] = 0;		//init above-threshold counter
						blw_thr_cnt[adc_chnl] = 0;		//init below-threshold counter
						trig_state[adc_chnl] = READY;
					}
				}
			}
		}
	}
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called when a MIDI package has been received
/////////////////////////////////////////////////////////////////////////////
void APP_MIDI_NotifyPackage(mios32_midi_port_t port, mios32_midi_package_t midi_package)
{
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called before the shift register chain is scanned
/////////////////////////////////////////////////////////////////////////////
void APP_SRIO_ServicePrepare(void)
{
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called after the shift register chain has been scanned
/////////////////////////////////////////////////////////////////////////////
void APP_SRIO_ServiceFinish(void)
{
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called when a button has been toggled
// pin_value is 1 when button released, and 0 when button pressed
/////////////////////////////////////////////////////////////////////////////
void APP_DIN_NotifyToggle(u32 pin, u32 pin_value)
{
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called when an encoder has been moved
// incrementer is positive when encoder has been turned clockwise, else
// it is negative
/////////////////////////////////////////////////////////////////////////////
void APP_ENC_NotifyChange(u32 encoder, s32 incrementer)
{
}


/////////////////////////////////////////////////////////////////////////////
// This hook is called when a pot has been moved
/////////////////////////////////////////////////////////////////////////////
void APP_AIN_NotifyChange(u32 pin, u32 pin_value)
{
}
